﻿using RestSharp;
using Farakonesh.Logic.ICommonOperations;
using Farakonesh.Logic.Log;
using System;
using System.Collections.Generic;
using System.IO;
using System.Net;
using System.Net.Http;
using System.Threading;
using System.Threading.Tasks;
using static System.Collections.Specialized.BitVector32;
namespace Farakonesh.Logic.CommonOperations
{
    public class RestRequestHelper : IRestRequestHelper
    {
        /// <summary>
        /// 15 minute
        /// </summary>
        private const int timeOutService = 900000;
        private readonly ILog _logger;
        private readonly IHttpClientFactory _httpClientFactory;
        public RestRequestHelper(ILog logger, IHttpClientFactory httpClientFactory)
        {
            _logger = logger;
            _httpClientFactory = httpClientFactory;
        }

        private RestClient CreateClientWithBestTls(string baseUrl)
        {
            if (!Uri.TryCreate(baseUrl, UriKind.Absolute, out Uri validUri))
            {
                _logger.Information($"Url {baseUrl} is not valid", 5);
            }

            var options = new RestClientOptions
            {
                BaseUrl = validUri,
                Timeout = TimeSpan.FromMilliseconds(timeOutService),
                ConfigureMessageHandler = handler =>
                {
                    var httpHandler = new HttpClientHandler();
                    try
                    {
                        httpHandler.SslProtocols = System.Security.Authentication.SslProtocols.Tls13
                                                 | System.Security.Authentication.SslProtocols.Tls12;
                    }
                    catch (System.ComponentModel.Win32Exception ex)
                    {
                        _logger.Warning($"TLS 1.3 not supported on this machine, fallback to TLS 1.2. {ex.Message}");
                        httpHandler.SslProtocols = System.Security.Authentication.SslProtocols.Tls12;
                    }
                    catch (PlatformNotSupportedException ex)
                    {
                        _logger.Warning($"Platform doesn't support requested SslProtocols, fallback to TLS 1.2. {ex.Message}");
                        httpHandler.SslProtocols = System.Security.Authentication.SslProtocols.Tls12;
                    }

                    return httpHandler;
                }
            };

            return new RestClient(options);
        }

        private RestClient CreateClient(string baseUrl)
        {
            if (!Uri.TryCreate(baseUrl, UriKind.Absolute, out Uri validUri))
            {
                _logger.Information($"Url {baseUrl} is not valid", 5);
            }
            return new RestClient(new RestClientOptions
            {
                BaseUrl = validUri,
                Timeout = TimeSpan.FromMilliseconds(timeOutService),
            });
        }


        private RestRequest CreateRequest(Method method, Dictionary<string, string> headers = null)
        {
            var request = new RestRequest("", method);
            request.AddHeader("Accept", "application/json");

            if (headers != null)
            {
                foreach (var item in headers)
                {
                    request.AddHeader(item.Key, item.Value);
                }
            }

            return request;
        }
        public async Task<RestResponse<T>> executeAsync<T>(string action, object body
            , Method method, CancellationToken cancellationToken, Dictionary<string, string> headers = null)
        {
            var requestClient = CreateClient(action);
            var request = CreateRequest(method, headers);

            request.AddBody(body, "application/json");

            var response = await requestClient.ExecuteAsync<T>(request, cancellationToken);
            _logger.Information($"Request to {action} returned {response.StatusCode} with response: {response.ObjToJson()}", 3);
            return response;
        }

        public async Task<RestResponse<T>> executeGraphqlAsync<T>(string action, string body
       , CancellationToken cancellationToken, Dictionary<string, string> headers = null)
        {
            var requestClient = CreateClient(action);
            var request = CreateRequest(Method.Post, headers);

            request.AddStringBody(body, DataFormat.Json);
            var response = await requestClient.ExecuteAsync<T>(request, cancellationToken);
            _logger.Information($"Request to {action} returned {response.StatusCode} with response: {response.ObjToJson()}", 3);

            return response;
        }


        public async Task<RestResponse<T>> executeAsyncMultiPart<T>(string baseUrl, string api, string url
           , Method method, CancellationToken cancellationToken, Dictionary<string, string> headers = null)
        {
            Uri baseApiUrl = new Uri(baseUrl);
            var options = new RestClientOptions
            {
                BaseUrl = baseApiUrl,
                Timeout = TimeSpan.FromMilliseconds(timeOutService)
            };
            var client = new RestClient(options);
            var request = new RestRequest(api, method);

            if (headers != null && headers.Count > 0)
            {
                foreach (var item in headers)
                {
                    request.AddHeader(item.Key, item.Value);
                }
            }

            var _client = _httpClientFactory.CreateClient();
            byte[] bytes = await _client.GetByteArrayAsync(url);

            request.AlwaysMultipartFormData = true;
            var fileExt = Path.GetExtension(url);
            string extPlain = "";
            switch (fileExt)
            {
                case ".jpg":
                    extPlain = "image/jpeg";
                    break;
                case ".png":
                    extPlain = "image/png";
                    break;
                default:
                    break;
            }
            string fileName = Path.GetFileName(url);
            request.AddFile("file", bytes, fileName, extPlain);
            var response = await client.ExecuteAsync<T>(request, cancellationToken);
            _logger.Information($"Request to {baseUrl + api} returned {response.StatusCode} with response: {response.Content.ObjToJson()}", 3);

            return response;
        }
        public async Task<RestResponse<T>> executeAsyncGet<T>(string action, Dictionary<string, string> parameters
        , CancellationToken cancellationToken, Dictionary<string, string> headers = null)
        {
            var requestClient = CreateClient(action);
            ServicePointManager.ServerCertificateValidationCallback +=
       (sender, certificate, chain, sslPolicyErrors) => true;
            var request = CreateRequest(Method.Get, headers);

            if (parameters != null && parameters.Count > 0)
            {
                foreach (var item in parameters)
                {
                    request.AddParameter(item.Key, item.Value);
                }
            }
            var response = await requestClient.ExecuteAsync<T>(request, cancellationToken);

            _logger.Information($"Request to {action} returned {response.StatusCode} with response: {response.ObjToJson()}", 3);

            return response;
        }

        public async Task<RestResponse<T>> executeAsyncGet<T>(string action, Dictionary<string, object> parameters
     , CancellationToken cancellationToken, Dictionary<string, string> headers = null)
        {
            var requestClient = CreateClient(action);
            var request = CreateRequest(Method.Get, headers);

            if (parameters != null && parameters.Count > 0)
            {
                foreach (var item in parameters)
                {
                    request.AddParameter(item.Key, item.Value, ParameterType.QueryString);
                }
            }


            var response = await requestClient.ExecuteAsync<T>(request, cancellationToken);
            _logger.Information($"Request to {action} returned {response.StatusCode} with response: {response.ObjToJson()}", 3);

            return response;
        }

        public async Task<RestResponse<T>> executeAsyncDelete<T>(string action, Dictionary<string, object> parameters
 , CancellationToken cancellationToken, Dictionary<string, string> headers = null)
        {
            var requestClient = CreateClient(action);
            var request = CreateRequest(Method.Delete, headers);

            if (parameters != null && parameters.Count > 0)
            {
                foreach (var item in parameters)
                {
                    request.AddParameter(item.Key, item.Value, ParameterType.QueryString);
                }
            }


            var response = await requestClient.ExecuteAsync<T>(request, cancellationToken);
            _logger.Information($"Request to {action} returned {response.StatusCode} with response: {response.ObjToJson()}", 3);

            return response;
        }

    }
}
