﻿using Microsoft.AspNetCore.Http;
using Microsoft.Extensions.Configuration;
using NLog;
using NLog.Web;
using System;
using System.Data.SqlClient;
using System.Threading.Tasks;

namespace Farakonesh.Logic.Log
{
    public class Log : ILog
    {
        private readonly string _connectionString;
        private static readonly Logger logger = NLogBuilder.ConfigureNLog("nlog.config").GetCurrentClassLogger();

        public Log(IConfiguration configuration)
        {
            _connectionString = configuration.GetConnectionString("cnsLogServer");
        }

        public void Information(string message, int logType)
        {
            logger.WithProperty("logType", logType).Info(message);
        }

        public void Information(string message)
        {
            logger.Info(message);
        }

        public void Warning(string message)
        {
            logger.WithProperty("logType", 4).Warn(message);
        }

        public void Debug(string message)
        {
            logger.Debug(message);
        }

        public void Error(string message)
        {
            logger.WithProperty("logType", 5).Error(message);
        }

        public void Add(Exception e, string customMessage = null)
        {
            string text = string.Format("{0}\nException: {1}\nStackTrace: {2}\n",
                customMessage ?? "", e.Message, e.StackTrace);
            this.Add(text);
        }

        public void Add(string message)
        {
            Information(message);
        }

        public async Task<long> LogErrorAndGetIdAsync(string message, string exception,
            string stackTrace, string url, int logType = 5)
        {
            await using var connection = new SqlConnection(_connectionString);
            await connection.OpenAsync();

            var command = new SqlCommand(@"
        INSERT INTO Logs (SaveDate, Message,Url, Level, Exception, StackTrace, Logger, LogType)
        OUTPUT INSERTED.Id
        VALUES (@SaveDate, @Message,@Url, @Level, @Exception, @StackTrace, @Logger, @LogType)", connection);

            command.Parameters.AddWithValue("@SaveDate", DateTime.Now);
            command.Parameters.AddWithValue("@Message", message);
            command.Parameters.AddWithValue("@Url", url);
            command.Parameters.AddWithValue("@Level", "Error");
            command.Parameters.AddWithValue("@Exception", exception ?? (object)DBNull.Value);
            command.Parameters.AddWithValue("@StackTrace", stackTrace ?? (object)DBNull.Value);
            command.Parameters.AddWithValue("@Logger", "ExceptionMiddleware");
            command.Parameters.AddWithValue("@LogType", logType);

            var id = (long)await command.ExecuteScalarAsync();
            return id;
        }
    }
}
