﻿using Microsoft.AspNetCore.Authentication.JwtBearer;
using Microsoft.AspNetCore.Http;
using Microsoft.Extensions.Configuration;
using Microsoft.IdentityModel.Tokens;
using Newtonsoft.Json.Linq;
using Farakonesh.Shared.Exceptions;
using Farakonesh.Logic.ISecurity;
using System;
using System.Collections.Generic;
using System.IdentityModel.Tokens.Jwt;
using System.Linq;
using System.Security.Claims;
using System.Text;
using System.Threading.Tasks;
using Farakonesh.Shared.Helpers;

namespace Farakonesh.Logic.Security
{
    public class TokenUserService : ITokenUserService
    {
        private readonly IConfiguration _configuration;
        public TokenUserService(IConfiguration configuration)
        {
            _configuration = configuration;
        }

        public UserInfo getUserInfo()
        {
            var userModel = getData(getJWTToken());
            if (userModel == null)
                return new UserInfo();

            return userModel;
        }

        public UserInfo getUserRefreshInfo(string token)
        {
            var userModel = getDataRefresh(token);
            if (userModel == null)
                return new UserInfo();

            return userModel;
        }

        private UserInfo getData(string token)
        {
            if(string.IsNullOrWhiteSpace(token))
            {
                throw new UserSignatureException("خطا در احراز هویت ، لطفا مجددا وارد برنامه شوید");
            }

            var validateToken = getPrincipal(token);

            if (validateToken != null)
            {
                var model = new UserInfo();
                if (validateToken.Identity != null && !string.IsNullOrWhiteSpace(validateToken.Identity.Name))
                {
                    model.Token = validateToken.Identity.Name;
                    var UserId = getValueFromClaim(validateToken.Claims, "UserId");
                    if (UserId != null)
                        model.UserId = UserId;
                    return model;
                }
            }

            return null;
        }

        private UserInfo getDataRefresh(string token)
        {
            if(string.IsNullOrWhiteSpace(token))
            {
                throw new UserSignatureException("خطا در احراز هویت ، لطفا مجددا وارد برنامه شوید");
            }
            var validateToken = getPrincipal(token);

            if (validateToken != null)
            {
                var model = new UserInfo();
                if (validateToken.Identity != null && !string.IsNullOrWhiteSpace(validateToken.Identity.Name))
                {
                    model.RefreshToken = validateToken.Identity.Name;
                    var UserId = getValueFromClaim(validateToken.Claims, "UserId");
                    if (UserId != null)
                        model.UserId = UserId;
                    return model;
                }
            }

            return null;
        }

        private ClaimsPrincipal getPrincipal(string token)
        {
            var tokenHandler = new JwtSecurityTokenHandler();
            var jwtToken = tokenHandler.ReadToken(token) as JwtSecurityToken;

            if (jwtToken == null)
                return null;

            var symmetricKey = Encoding.UTF8.GetBytes(_configuration.GetValue<string>("JWT:Key"));
            var validationParameters = new TokenValidationParameters()
            {
                RequireExpirationTime = false,
                ValidateIssuer = true,
                ValidateAudience = false,
                ValidateLifetime = false,
                ValidIssuer = _configuration.GetValue<string>("JWT:Issuer"),
                IssuerSigningKey = new SymmetricSecurityKey(symmetricKey)
            };

            var principal = tokenHandler.ValidateToken(token, validationParameters, out _);

            return principal;
        }

        private string getValueFromClaim(IEnumerable<Claim> claims, string name)
        {
            try
            {
                var item = claims.FirstOrDefault(a => a.Type.ToLower().Equals(name.ToLower()));
                return (item.Value);
            }
            catch (Exception)
            {
                return null;
            }
        }
        public string getJWTToken()
        {
            IHttpContextAccessor ctx = new HttpContextAccessor();
            if (ctx.HttpContext == null)
            {
                throw new RequestException("خطا در ارسال درخواست ، اطلاعات کلاینت جاری یافت نشد");
            }
            var headers = ctx.HttpContext.Request.Headers["Authorization"];
            if (headers.Count == 0)
                //در صورتی که نتوانستی اطلاعات را بدست بیاوری، با حروف کوچک امتحان کن
                headers = ctx.HttpContext.Request.Headers["authorization"];

            if (headers.Count == 0)
                return null;

            var header = headers.FirstOrDefault();
            string token = "";
            if (header != null)
            {
                token = header.Replace("Bearer ", "").Replace("bearer ", "");
            }

            return (token);

        }



        /// <summary>
        /// بررسی کلیم های ارسالی جهت درج در توکن
        /// </summary>
        /// <param name="claims"></param>
        /// <param name="key"></param>
        /// <param name="issuer"></param>
        /// <param name="expireDate"></param>
        private void validation(Claim[] claims, DateTime expireDate)
        {
            if (claims == null || claims.Count() == 0)
                throw new RequestException("در تولید امضای کاربری خطایی اتفاق افتاده است ، لطفا در صورت تکرار خطا مراتب را به پشتیبانی سیستم اطلاع دهید");

            if (string.IsNullOrWhiteSpace(_configuration.GetValue<string>("JWT:Key")))
                throw new RequestException("در تولید امضای کاربری خطایی اتفاق افتاده است ، لطفا در صورت تکرار خطا مراتب را به پشتیبانی سیستم اطلاع دهید");

            if (string.IsNullOrWhiteSpace(_configuration.GetValue<string>("JWT:Issuer")))
                throw new RequestException("در تولید امضای کاربری خطایی اتفاق افتاده است ، لطفا در صورت تکرار خطا مراتب را به پشتیبانی سیستم اطلاع دهید");

            if (expireDate < DateTimeHelper.GetLocalTime())
                throw new RequestException("در تولید امضای کاربری خطایی اتفاق افتاده است ، لطفا در صورت تکرار خطا مراتب را به پشتیبانی سیستم اطلاع دهید");
        }

        /// <summary>
        /// تولید توکن با استفاده از پارامتر های ارسالی
        /// </summary>
        /// <param name="claims"></param>
        /// <param name="key"></param>
        /// <param name="issuer"></param>
        /// <param name="expireDate"></param>
        /// <returns></returns>
        public string generate(Claim[] claims, DateTime expireDate)
        {
            validation(claims, expireDate);

            var securityKey = new SymmetricSecurityKey(Encoding.UTF8.GetBytes(_configuration.GetValue<string>("JWT:Key")));
            var credentials = new SigningCredentials(securityKey, SecurityAlgorithms.HmacSha256);

            JwtSecurityToken token = new JwtSecurityToken
            (
                _configuration.GetValue<string>("JWT:Issuer"), 
                _configuration.GetValue<string>("JWT:Issuer"),
                claims,
                expires: expireDate,
                signingCredentials: credentials
            );

            var jwt = new JwtSecurityTokenHandler().WriteToken(token);
            return jwt;
        }
    }
}
