﻿using Farakonesh.Commands.Services;
using Farakonesh.Commands.Services.Order;
using Farakonesh.Logic.ICommonOperations;
using Farakonesh.Logic.IDatabase.IOrder;
using Farakonesh.Logic.IDatabase.IUser;
using Farakonesh.Logic.ISecurity;
using Farakonesh.Logic.IServices.IApp.IOrder;
using Farakonesh.Logic.IServices.ICache;
using Farakonesh.Models.Database;
using Farakonesh.Models.Database.StoredProcedures.App.Order;
using Farakonesh.Models.Database.StoredProcedures.App.Order.Orders;
using Farakonesh.Models.Database.StoredProcedures.App.User.Ticket;
using Farakonesh.Models.Database.StoredProcedures.App.User.User;
using Farakonesh.Shared.Enums;
using Farakonesh.Shared.Exceptions;
using Farakonesh.Shared.Helpers;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Channels;
using System.Threading.Tasks;

namespace Farakonesh.Logic.Services.App.Order
{
    public class AccountNumberService : IAccountNumberService
    {
        private readonly IAccountNumberContext _accountNumberContext;
        private readonly ITokenUserService _tokenUserService;
        private readonly IOrderContext _orderContext;
        private readonly IAutomaticOrderOperationService _automaticOrderOperationService;
        private readonly IRedisContextService _redisContextService;
        public AccountNumberService(IAccountNumberContext accountNumberContext,
            ITokenUserService tokenUserService
            , IAutomaticOrderOperationService automaticOrderOperationService, IOrderContext orderContext
            , IRedisContextService redisContextService)
        {
            _accountNumberContext = accountNumberContext;
            _tokenUserService = tokenUserService;
            _automaticOrderOperationService = automaticOrderOperationService;
            _orderContext = orderContext;
            _redisContextService = redisContextService;
        }
        public async Task<DBResult> GetAccountNumber(GetAccountNumberCommand command, CancellationToken cancellationToken)
        {
            string token = _tokenUserService.getUserInfo().Token;
            var order = await _orderContext.GetOrderByIdWithoutToken(new GetOrderByIdWithoutToken.Inputs
            {
                Code = command.Code,
                OrderId = command.OrderId
            }, cancellationToken);
            if (order == null || order.Data == null)
            {
                throw new LogicalException("فاکتور ثبت شده یافت نشد");
            }
            var account = await _accountNumberContext.GetAccountNumberActive(new GetAccountNumberActive.Inputs
            {
                Token = token,
                IsSmartPayment = command.IsSmartPayment,
                OrderId = command.OrderId,
            }, cancellationToken);
            if (account == null || account.Data == null || account.Data.AccountNumberId == null)
            {
                throw new LogicalException("شماره حساب فعالی جهت واریز وجه یافت نشد ، جهت پرداخت لطفا با پشتیبانی تماس بگیرید");
            }
            if (account.Data.UserId == null)
            {
                throw new LogicalException("شناسه شما یافت نشد ، لطفا برای حل مشکل با پشتیبانی تماس بگیرید");
            }
 
            var remainAmount = account.Data.FinalAmount - (order.Data.PaidAmount ?? 0);

            decimal? allowedAmount = 0;
            if (command.IsSmartPayment == true)
            {
                allowedAmount = await _automaticOrderOperationService.CheckScenario(new CheckScenarioCommand
                {
                    OrderType = account.Data.OrderType,
                    PaidAmount = order.Data.PaidAmount,
                    FinalAmount = account.Data.FinalAmount,
                    FinalWageAmount = order.Data.FinalWageAmount,
                    PaymentMethod = PayMentMethod.SmartDeposit,
                    DiscountedFinalAmount = order.Data.DiscountedFinalAmount,
                    ModuleType = order.Data.SaveLocation,
                }, cancellationToken);
                if (allowedAmount == null || allowedAmount <= 0)
                {
                    allowedAmount = remainAmount;
                }
            }



            var responseFreeOrder = await _automaticOrderOperationService.doCompeleteFreeOrder(new DoCompeleteFreeOrderCommand
            {
                Code = command.Code,
                OrderId = command.OrderId,
                model = order.Data,
                CheckToken = true
            }, cancellationToken);
            if (responseFreeOrder != null)
                return responseFreeOrder;

            var settingOrder = await _redisContextService.getSettingOrderAsync(order.Data.OrderType, token, cancellationToken);

            if ((settingOrder.IsActiveAccountByAccountSmart == null || settingOrder.IsActiveAccountByAccountSmart == false)
                && (settingOrder.IsActiveAccountByAccount == null || settingOrder.IsActiveAccountByAccount == false))
                throw new LogicalException("روش پرداخت حساب به حساب غیر فعال می باشد ، لطفا روش پرداخت دیگری را انتخاب کنید");
            account.Data.ShebaNumber = "IR" + account.Data.ShebaNumber;

            order.Data.FinalAmount = allowedAmount;
            return new DBResult()
            {
                Data = new
                {
                    account.Data?.CardNumber,
                    account.Data?.DepositNumber,
                    account.Data?.ShebaNumber,
                    account.Data?.AccountOwnerFullName,
                    account.Data?.OrderNumber,
                    account.Data?.Description,
                    account.Data?.DescriptionOnly,
                    account.Data?.IsSmartPayment,
                    order?.Data?.FinalAmount,
                    order?.Data?.PaidAmount,
                    order?.Data?.ExpireDate,
                    order?.Data?.DateNow,
                    account.Data?.BankAvatarId,
                    account.Data?.BankTitle,
                }
            };

        }
    }
}
