﻿using Farakonesh.Commands.Services;
using Farakonesh.Logic.CommonOperations;
using Farakonesh.Logic.IDatabase.idbo;
using Farakonesh.Logic.IDatabase.IOrder;
using Farakonesh.Logic.IDatabase.IQueue;
using Farakonesh.Logic.ISecurity;
using Farakonesh.Logic.IServices.IApp.IOrder;
using Farakonesh.Logic.IServices.IApp.IPPG;
using Farakonesh.Logic.IServices.ICache;
using Farakonesh.Logic.IServices.IRestRequest;
using Farakonesh.Models.API.JibitOnlinePay;
using Farakonesh.Models.Database;
using Farakonesh.Models.Database.StoredProcedures.App.dbo.setting;
using Farakonesh.Models.Database.StoredProcedures.App.Order;
using Farakonesh.Models.Database.StoredProcedures.App.Order.Orders;
using Farakonesh.Models.Database.StoredProcedures.App.Order.Transaction;
using Farakonesh.Models.Database.StoredProcedures.App.Queue;
using Farakonesh.Shared.Enums;
using Farakonesh.Shared.Exceptions;
using Farakonesh.Shared.Helpers;
using Microsoft.AspNetCore.Http;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace Farakonesh.Logic.Services.App.PPG
{
    public class JibitService : IJibitService
    {
        private readonly IAutomaticOrderOperationService _IAutomaticOrderOperationService;
        private readonly IOrderContext _IOrderContext;
        private readonly ISettingContext _ISettingContext;
        private readonly ITransActionContext _ITransActionContext;
        private readonly IQueueContext _IQueueContext;
        private readonly IHttpRequestJibitPPG _httpRequestJibitPPG;
        private readonly ITokenUserService _tokenUserService;
        private readonly IRedisContextService _redisContextService;
        private readonly IHttpContextAccessor _httpContextAccessor;
        public JibitService(IAutomaticOrderOperationService IAutomaticOrderOperationService
            , ISettingContext iSettingContext, IOrderContext iOrderContext
            , ITransActionContext iTransActionContext
            , IQueueContext iQueueContext, IHttpRequestJibitPPG httpRequestJibitPPG,
            ITokenUserService tokenUserService, IRedisContextService redisContextService,
            IHttpContextAccessor httpContextAccessor)
        {
            _IAutomaticOrderOperationService = IAutomaticOrderOperationService;
            _ISettingContext = iSettingContext;
            _IOrderContext = iOrderContext;
            _ITransActionContext = iTransActionContext;
            _IQueueContext = iQueueContext;
            _httpRequestJibitPPG = httpRequestJibitPPG;
            _tokenUserService = tokenUserService;
            _redisContextService = redisContextService;
            _httpContextAccessor = httpContextAccessor;
        }
        private async Task<string> GetJibitTokenAsync(bool getLastToken, CancellationToken cancellationToken)
        {
            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGJibit, cancellationToken);

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            string jibitToken = "";
            if (getLastToken)
            {
                var dbResult = await _IQueueContext.Get_Last_Token(new Get_Last_Token.Inputs { TokenType = (int)TokenType.JibitPPG }, cancellationToken);
                jibitToken = dbResult.Data.JibitTokenPPG;
            }

            if (string.IsNullOrWhiteSpace(jibitToken))
            {
                var responseJibit = await _httpRequestJibitPPG.getTokenJibitPPG(webServiceInfo.RealApiKey, webServiceInfo.RealPassword, webServiceInfo.BaseUrlService, cancellationToken);
                if (responseJibit == null || string.IsNullOrWhiteSpace(responseJibit.accessToken))
                    throw new ExternalServiceException("خطا در دریافت امضای درگاه پرداخت");

                jibitToken = responseJibit.accessToken;
                await _IQueueContext.Insert_Token(new Insert_Token.Inputs { JibitTokenPPG = responseJibit.accessToken }, cancellationToken);
            }

            return jibitToken;
        }

        public async Task<DBResult> startPay(StartPayOrderCommand command, CancellationToken cancellationToken)
        {
            string token = _tokenUserService.getUserInfo().Token;
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGJibit, cancellationToken);

            if (webServiceInfo == null)
                throw new LogicalException("درگاه پرداخت غیر فعال می باشد ، لطفا درگاه پرداخت دیگری را انتخاب کنید");

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var dbResultOrder = await _IOrderContext.GetOrderByIdWithoutToken(new GetOrderByIdWithoutToken.Inputs
            {
                OrderId = command.OrderId,
                Code = command.Code
            }, cancellationToken);
            if (dbResultOrder == null || dbResultOrder.Data == null)
            {
                throw new LogicalException(MessageHelper.NotFoundOrder);
            }
            var allowedAmount = await _IAutomaticOrderOperationService.CheckScenario(new CheckScenarioCommand
            {
                OrderType = dbResultOrder.Data.OrderType,
                PaidAmount = dbResultOrder.Data.PaidAmount,
                FinalAmount = dbResultOrder.Data.FinalAmount,
                FinalWageAmount = dbResultOrder.Data.FinalWageAmount,
                PaymentMethod = PayMentMethod.OnlinePayment,
                DiscountedFinalAmount = dbResultOrder.Data.DiscountedFinalAmount,
                ModuleType = dbResultOrder.Data.SaveLocation
            }, cancellationToken);

            var responseFreeOrder = await _IAutomaticOrderOperationService.doCompeleteFreeOrder(new DoCompeleteFreeOrderCommand
            {
                Code = command.Code,
                OrderId = command.OrderId,
                model = dbResultOrder.Data,
                CheckToken = true
            }, cancellationToken);
            if (responseFreeOrder != null)
                return responseFreeOrder;

            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);

            var localIp = _httpContextAccessor.HttpContext?.Connection.LocalIpAddress;
            var isExternal = await IpCheckHelper.IsExternalServerAsync(setting.ExternalServerIp, new IpifyPublicIpProvider(), localIp);
            if (isExternal)
            {
                throw new LogicalException(MessageHelper.ExrenalServerErrorPayment);
            }
            var settingOrder = await _redisContextService.getSettingOrderAsync(dbResultOrder.Data.OrderType,
                token, cancellationToken);


            var dbResultBankAccount = await _IAutomaticOrderOperationService.checkBankAccountClientPortal(
             new CheckBankAccountClientPortalCommand
             {
                 BankAccountId = command.BankAccountId,
                 CheckBankAccount = settingOrder.CheckBankAccount,
                 Token = token
             }, cancellationToken);

            await _IAutomaticOrderOperationService.CheckIpClientPortal(cancellationToken);

            var jibitToken = await GetJibitTokenAsync(true, cancellationToken);

            string refrenceNumber = Guid.NewGuid().ToString();
            var maximumAmountOnlinePort = Convert.ToDecimal(settingOrder.MaximumAmountOnlinePort);
            var minimumAmountOnlinePort = Convert.ToDecimal(settingOrder.MinimumAmountOnlinePort);
            var finalAmountToman = (allowedAmount != null && allowedAmount > 0) ? allowedAmount : dbResultOrder.Data.Discount != null && dbResultOrder.Data.Discount > 0 ? dbResultOrder.Data.DiscountedFinalAmount : dbResultOrder.Data.FinalAmount;
            finalAmountToman = finalAmountToman.Value.ValidatePaymentAmountMultiPayment(maximumAmountOnlinePort, minimumAmountOnlinePort,
                dbResultOrder.Data.PaidAmount, settingOrder.IsActiveOnlinePaySingle,
                settingOrder.IsActiveOnlinePayMultiple);

            var responsePurchasesJibitPPG = await _httpRequestJibitPPG.purchaseJibitPPG(new RequestPurchasesJibitPPG
            {
                amount = finalAmountToman.toRial(),
                callbackUrl = (string)config["callback"] + ("?returnUrl=" + command.ReturnUrl).Trim(),
                clientReferenceNumber = refrenceNumber,
                checkPayerNationalCode = false,
                payerNationalCode = settingOrder.CheckBankAccount ? dbResultBankAccount?.Data?.FirstOrDefault()?.NationalCode : null,
                payerCardNumber = settingOrder.CheckBankAccount ? dbResultBankAccount?.Data?.FirstOrDefault()?.CardNumber : null,
                switching = new RequestPurchasesJibitSwitchingPPG { autoSwitching = (string)config["smartSwitch"] == "1" ? true : false, terminalIds = !string.IsNullOrWhiteSpace((string)config["terminal"]) ? config["terminal"].ToString().Split(",").ToList() : new List<string>() },
            }, jibitToken, webServiceInfo.BaseUrlService, cancellationToken);
            responsePurchasesJibitPPG.handlerRequestPPG(false);

            if (responsePurchasesJibitPPG.errors != null && responsePurchasesJibitPPG.errors.Count > 0 && responsePurchasesJibitPPG.errors[0].code == "token.verification_failed")
            {
                jibitToken = await GetJibitTokenAsync(false, cancellationToken);
                responsePurchasesJibitPPG = await _httpRequestJibitPPG.purchaseJibitPPG(new RequestPurchasesJibitPPG
                {
                    amount = finalAmountToman.toRial(),
                    callbackUrl = (string)config["callback"] + ("?returnUrl=" + command.ReturnUrl).Trim(),
                    clientReferenceNumber = refrenceNumber,
                    checkPayerNationalCode = false,
                    payerCardNumber = settingOrder.CheckBankAccount ? dbResultBankAccount.Data.FirstOrDefault().CardNumber : null,
                    payerNationalCode = settingOrder.CheckBankAccount ? dbResultBankAccount.Data.FirstOrDefault().NationalCode : null,
                    switching = new RequestPurchasesJibitSwitchingPPG { autoSwitching = (string)config["smartSwitch"] == "1" ? true : false, terminalIds = !string.IsNullOrWhiteSpace((string)config["terminal"]) ? config["terminal"].ToString().Split(",").ToList() : new List<string>() },
                }, jibitToken, webServiceInfo.BaseUrlService, cancellationToken);

            }

            responsePurchasesJibitPPG.handlerRequestPPG(true);

            await _ITransActionContext.Insert_Transaction(new Insert_Transaction.Inputs
            {
                Token = token,
                OrderId = command.OrderId,
                PaidAmount = finalAmountToman,
                PaidCode = responsePurchasesJibitPPG.purchaseId.ToString(),
                TrackingCode = refrenceNumber,
                MaximummPrice = maximumAmountOnlinePort,
                OnlinePortal = (int)OnlinePortal.Jibit,
                Code = command.Code
            }, cancellationToken);

            return new DBResult(new { redirectUrl = responsePurchasesJibitPPG.pspSwitchingUrl, IsFree = false }, 1);
        }

        public async Task<DBResult> startPayPublic(StartPayPublicOrderCommand command, CancellationToken cancellationToken)
        {
            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGJibit, cancellationToken);

            if (webServiceInfo == null)
                throw new LogicalException("درگاه پرداخت غیر فعال می باشد ، لطفا درگاه پرداخت دیگری را انتخاب کنید");

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var dbResultOrder = await _IOrderContext.GetOrderByIdWithoutToken(new GetOrderByIdWithoutToken.Inputs { Code = command.Code, OrderNumber = command.OrderNumber }, cancellationToken);
            if (dbResultOrder == null || dbResultOrder.Data == null)
            {
                throw new LogicalException(MessageHelper.NotFoundOrder);
            }
            command.OrderId = dbResultOrder.Data.OrderId;
            var settingOrder = await _redisContextService.getSettingOrderWithoutTokenAsync(dbResultOrder.Data.OrderType,
                command.OrderId, cancellationToken);

            var responseFreeOrder = await _IAutomaticOrderOperationService.doCompeleteFreeOrder(new DoCompeleteFreeOrderCommand
            {
                Code = command.Code,
                OrderId = command.OrderId,
                model = dbResultOrder.Data,
                CheckToken = false
            }, cancellationToken);
            if (responseFreeOrder != null)
                return responseFreeOrder;

            var allowedAmount = await _IAutomaticOrderOperationService.CheckScenario(new CheckScenarioCommand
            {
                OrderType = dbResultOrder.Data.OrderType,
                PaidAmount = dbResultOrder.Data.PaidAmount,
                FinalAmount = dbResultOrder.Data.FinalAmount,
                FinalWageAmount = dbResultOrder.Data.FinalWageAmount,
                PaymentMethod = PayMentMethod.OnlinePayment,
                DiscountedFinalAmount = dbResultOrder.Data.DiscountedFinalAmount,
                ModuleType = dbResultOrder.Data.SaveLocation
            }, cancellationToken);

            var dbResultBankAccount = await _IAutomaticOrderOperationService.checkBankAccountClientPortalWithoutToken(
           new CheckBankAccountClientPortalCommand
           {
               BankAccountId = command.BankAccountId,
               CheckBankAccount = settingOrder.CheckBankAccount
           }, cancellationToken);


            await _IAutomaticOrderOperationService.CheckIpClientPortal(cancellationToken);

            var jibitToken = await GetJibitTokenAsync(true, cancellationToken);


            var responsePurchasesJibitPPG = new ResponsePurchasesJibitPPG();
            string refrenceNumber = Guid.NewGuid().ToString();

            var maximumAmountOnlinePort = Convert.ToDecimal(settingOrder.MaximumAmountOnlinePort);
            var minimumAmountOnlinePort = Convert.ToDecimal(settingOrder.MinimumAmountOnlinePort);
            var finalAmountToman = (allowedAmount != null && allowedAmount > 0) ? allowedAmount : dbResultOrder.Data.Discount != null && dbResultOrder.Data.Discount > 0 ? dbResultOrder.Data.DiscountedFinalAmount : dbResultOrder.Data.FinalAmount;
            finalAmountToman = finalAmountToman.Value.ValidatePaymentAmountMultiPayment(maximumAmountOnlinePort, minimumAmountOnlinePort,
    dbResultOrder.Data.PaidAmount, settingOrder.IsActiveOnlinePaySingle,
    settingOrder.IsActiveOnlinePayMultiple);

            responsePurchasesJibitPPG = await _httpRequestJibitPPG.purchaseJibitPPG(new RequestPurchasesJibitPPG
            {
                amount = finalAmountToman.toRial(),
                callbackUrl = (string)config["callback"],
                clientReferenceNumber = refrenceNumber,
                checkPayerNationalCode = false,
                payerNationalCode = settingOrder.CheckBankAccount ? dbResultBankAccount?.Data?.FirstOrDefault()?.NationalCode : null,
                payerCardNumber = settingOrder.CheckBankAccount ? dbResultBankAccount?.Data?.FirstOrDefault()?.CardNumber : null,
                switching = new RequestPurchasesJibitSwitchingPPG { autoSwitching = (string)config["smartSwitch"] == "1" ? true : false, terminalIds = !string.IsNullOrWhiteSpace((string)config["terminal"]) ? config["terminal"].ToString().Split(",").ToList() : new List<string>() },
            }, jibitToken, webServiceInfo.BaseUrlService, cancellationToken);

            responsePurchasesJibitPPG.handlerRequestPPG(false);

            if (responsePurchasesJibitPPG.errors != null && responsePurchasesJibitPPG.errors.Count > 0 && responsePurchasesJibitPPG.errors[0].code == "token.verification_failed")
            {
                jibitToken = await GetJibitTokenAsync(false, cancellationToken);
                responsePurchasesJibitPPG = await _httpRequestJibitPPG.purchaseJibitPPG(new RequestPurchasesJibitPPG
                {
                    amount = finalAmountToman.toRial(),
                    callbackUrl = (string)config["callback"],
                    clientReferenceNumber = refrenceNumber,
                    checkPayerNationalCode = false,
                    payerCardNumber = settingOrder.CheckBankAccount ? dbResultBankAccount?.Data?.FirstOrDefault()?.CardNumber : null,
                    payerNationalCode = settingOrder.CheckBankAccount ? dbResultBankAccount?.Data?.FirstOrDefault()?.NationalCode : null,
                    switching = new RequestPurchasesJibitSwitchingPPG { autoSwitching = (string)config["smartSwitch"] == "1" ? true : false, terminalIds = !string.IsNullOrWhiteSpace((string)config["terminal"]) ? config["terminal"].ToString().Split(",").ToList() : new List<string>() },
                }, jibitToken, webServiceInfo.BaseUrlService, cancellationToken);


            }

            responsePurchasesJibitPPG.handlerRequestPPG(true);

            await _ITransActionContext.Insert_Transaction_WithoutToken(new Insert_Transaction_WithoutToken.Inputs
            {
                UserId = dbResultOrder.Data.UserId,
                OrderId = command.OrderId,
                PaidAmount = finalAmountToman,
                PaidCode = responsePurchasesJibitPPG.purchaseId.ToString(),
                TrackingCode = refrenceNumber,
                MaximummPrice = maximumAmountOnlinePort,
                OnlinePortal = (int)OnlinePortal.Jibit,
                Code = command.Code
            }, cancellationToken);
            return new DBResult(new { redirectUrl = responsePurchasesJibitPPG.pspSwitchingUrl, IsFree = false }, 1);
        }

        public async Task<DBResult> verifyPay(VerifyPayOrderCommand command, CancellationToken cancellationToken)
        {
            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGJibit, cancellationToken);

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            int? statusOrder = null;

            var maximumAmountOnlinePort = Convert.ToDecimal(setting.MaximumAmountOnlinePort);
            var dbResultCheckTransaction = await _ITransActionContext.Check_Duplicate_Transaction(new Check_Duplicate_Transaction.Inputs
            {
                PaidCode = command.PurchaseId,
                Status = command.Status
            }, cancellationToken);

            if (dbResultCheckTransaction != null && dbResultCheckTransaction.Data != null && dbResultCheckTransaction.Data.IsDuplicate == true)
            {
                return _IAutomaticOrderOperationService.getPPGResult(new GetDbResultPPGCommand
                {
                    Status = "SUCCESSFUL",
                    Setting = setting,
                    FinalAmount = dbResultCheckTransaction.Data.PaidAmount,
                    CompanyId = dbResultCheckTransaction.Data.CompanyId,
                    OrderId = dbResultCheckTransaction.Data.OrderId,
                    TrackingCode = dbResultCheckTransaction.Data.TrackingCode,
                    TransactionCode = dbResultCheckTransaction.Data.TransactionCode,
                    PaidCode = dbResultCheckTransaction.Data.PaidCode
                });
            }

            if (command.Status == "SUCCESSFUL")
            {

                var jibitToken = await GetJibitTokenAsync(true, cancellationToken);

                var responsePurchasesVerifyJibitPPG = await _httpRequestJibitPPG.verifyJibitPPG(command.PurchaseId
                    , jibitToken
                    , webServiceInfo.BaseUrlService, cancellationToken);

                if (responsePurchasesVerifyJibitPPG == null)
                {
                    var dbResultUpdateStatus = await _IOrderContext.Update_Order_Status_WithouToken(new Update_Order_Status_WithouToken.Inputs
                    {
                        OrderId = dbResultCheckTransaction.Data.OrderId,
                        Status = (int)OrderStatus.NotConfirmed
                    }, cancellationToken);
                    throw new ExternalServiceException("خطا در دریافت نتیجه ی درگاه پرداخت ، در صورت تکرار خطا مراتب را به پشتیبانی سیستم اطلاع دهید");
                }

                if (responsePurchasesVerifyJibitPPG != null && responsePurchasesVerifyJibitPPG.errors != null && responsePurchasesVerifyJibitPPG.errors.Count > 0 && responsePurchasesVerifyJibitPPG.errors[0].code == "purchase.not_found")
                {
                    var dbResultUpdateStatus = await _IOrderContext.Update_Order_Status_WithouToken(new Models.Database.StoredProcedures.App.Order.Update_Order_Status_WithouToken.Inputs
                    {
                        OrderId = dbResultCheckTransaction.Data.OrderId,
                        Status = (int)OrderStatus.NotConfirmed
                    }, cancellationToken);
                    throw new ExternalServiceException("کد پرداخت ارسال شده اشتباه است ، در صورت تکرار خطا مراتب را به پشتیبانی سیستم اطلاع دهید");
                }


                if (responsePurchasesVerifyJibitPPG != null && responsePurchasesVerifyJibitPPG.errors != null && responsePurchasesVerifyJibitPPG.errors.Count > 0 && responsePurchasesVerifyJibitPPG.errors[0].code == "token.verification_failed")
                {
                    jibitToken = await GetJibitTokenAsync(false, cancellationToken);
                    responsePurchasesVerifyJibitPPG = await _httpRequestJibitPPG.verifyJibitPPG(command.PurchaseId
                        , jibitToken, webServiceInfo.BaseUrlService, cancellationToken);

                    if (responsePurchasesVerifyJibitPPG == null)
                    {
                        var dbResultUpdateStatus = await _IOrderContext.Update_Order_Status_WithouToken(new Update_Order_Status_WithouToken.Inputs
                        {
                            OrderId = dbResultCheckTransaction.Data.OrderId,
                            Status = (int)OrderStatus.NotConfirmed
                        }, cancellationToken);
                        throw new ExternalServiceException("خطا در دریافت نتیجه ی درگاه پرداخت ، در صورت تکرار خطا مراتب را به پشتیبانی سیستم اطلاع دهید");
                    }

                    if (responsePurchasesVerifyJibitPPG != null && responsePurchasesVerifyJibitPPG.errors != null && responsePurchasesVerifyJibitPPG.errors.Count > 0 && responsePurchasesVerifyJibitPPG.errors[0].code == "purchase.not_found")
                    {
                        var dbResultUpdateStatus = await _IOrderContext.Update_Order_Status_WithouToken(new Update_Order_Status_WithouToken.Inputs
                        {
                            OrderId = dbResultCheckTransaction.Data.OrderId,
                            Status = (int)OrderStatus.NotConfirmed
                        }, cancellationToken);
                        throw new ExternalServiceException("کد پرداخت ارسال شده اشتباه است ، در صورت تکرار خطا مراتب را به پشتیبانی سیستم اطلاع دهید");
                    }
                }

                if (responsePurchasesVerifyJibitPPG != null && responsePurchasesVerifyJibitPPG.errors != null && responsePurchasesVerifyJibitPPG.errors.Count > 0)
                {
                    var dbResultUpdateStatus = await _IOrderContext.Update_Order_Status_WithouToken(new Update_Order_Status_WithouToken.Inputs
                    {
                        OrderId = dbResultCheckTransaction.Data.OrderId,
                        Status = (int)OrderStatus.NotConfirmed
                    }, cancellationToken);
                    throw new ExternalServiceException("خطا در ارسال درخواست به درگاه آنلاین ، در صورت کسر وجه مبلغ پرداخت شده طی 24 ساعت به حساب شما برگشت داده خواهد شد");
                }

                if (responsePurchasesVerifyJibitPPG.status == "SUCCESSFUL")
                {
                    var dbResultUpdateTransaction = await _ITransActionContext.Update_Transaction(new Update_Transaction.Inputs
                    {
                        TransactionCode = command.TransactionCode,
                        PaidAmount = Convert.ToDecimal(command.PaidAmount) / 10,
                        PaidCode = command.PurchaseId,
                        Status = command.Status,
                        TrackingCode = command.TrackingCode,
                        MaximummPrice = maximumAmountOnlinePort
                    }, cancellationToken);
                    statusOrder = dbResultUpdateTransaction.Data.StatusOrder;


                    if (dbResultUpdateTransaction.Data.StatusOrder != null && dbResultUpdateTransaction.Data.StatusOrder == 192)
                    {
                        return await _IAutomaticOrderOperationService.DoCompeleteScenario(new DoCompeleteScenarioCommand
                        {
                            OrderNumber = dbResultUpdateTransaction.Data.OrderNumber,
                            Email = dbResultUpdateTransaction.Data.Email,
                            Mobile = dbResultUpdateTransaction.Data.Mobile,
                            OrderId = dbResultUpdateTransaction.Data.OrderId,
                            OrderType = dbResultUpdateTransaction.Data.OrderType,
                            RemainingAmount = dbResultUpdateTransaction.Data.RemainingAmount,
                            StatusOrder = dbResultUpdateTransaction.Data.StatusOrder,
                            FinalAmount = dbResultUpdateTransaction.Data.FinalAmount,
                            Status = responsePurchasesVerifyJibitPPG.status,
                            CompanyId = dbResultUpdateTransaction.Data.CompanyOwnerId,
                            Setting = setting,
                            PaidCode = dbResultUpdateTransaction.Data.PaidCode,
                            TransactionCode = dbResultUpdateTransaction.Data.TransactionCode,
                            TrackingCode = dbResultUpdateTransaction.Data.TrackingCode,
                            ModuleType = dbResultUpdateTransaction.Data.SaveLocation
                        }, cancellationToken);
                    }
                    return await _IAutomaticOrderOperationService.doCompelete(new DoCompeleteCommand
                    {
                        Email = dbResultUpdateTransaction.Data.Email,
                        Mobile = dbResultUpdateTransaction.Data.Mobile,
                        OrderId = dbResultUpdateTransaction.Data.OrderId,
                        OrderNumber = dbResultUpdateTransaction.Data.OrderNumber,
                        OrderType = dbResultUpdateTransaction.Data.OrderType,
                        PurchaseId = command.PurchaseId,
                        ValueType = dbResultUpdateTransaction.Data.ValueType,
                        Status = responsePurchasesVerifyJibitPPG.status,
                        StatusOrder = dbResultUpdateTransaction.Data.StatusOrder,
                        Commission = dbResultUpdateTransaction.Data.Commission,
                        EmailCompanyUser = dbResultUpdateTransaction.Data.EmailCompanyUser,
                        MobileCompanyUser = dbResultUpdateTransaction.Data.MobileCompanyUser,
                        TransactionType = dbResultUpdateTransaction.Data.TransactionType,
                        CooperRewardToman = dbResultUpdateTransaction.Data.CooperRewardToman,
                        NumberOfSuccessfulOrders = dbResultUpdateTransaction.Data.NumberOfSuccessfulOrders,
                        FinalAmount = dbResultUpdateTransaction.Data.FinalAmount.ToString(),
                        CompanyId = dbResultUpdateTransaction.Data.CompanyOwnerId,
                        Setting = setting,
                        PaidCode = dbResultUpdateTransaction.Data.PaidCode,
                        TransactionCode = dbResultUpdateTransaction.Data.TransactionCode,
                        TrackingCode = dbResultUpdateTransaction.Data.TrackingCode,
                    }, cancellationToken);

                }

                return _IAutomaticOrderOperationService.getPPGResult(new GetDbResultPPGCommand
                {
                    Status = responsePurchasesVerifyJibitPPG.status,
                    Setting = setting,
                    FinalAmount = dbResultCheckTransaction.Data.PaidAmount,
                    CompanyId = dbResultCheckTransaction.Data.CompanyId,
                    OrderId = dbResultCheckTransaction.Data.OrderId,
                    TrackingCode = dbResultCheckTransaction.Data.TrackingCode,
                    TransactionCode = dbResultCheckTransaction.Data.TransactionCode,
                    PaidCode = dbResultCheckTransaction.Data.PaidCode
                });
            }

            return _IAutomaticOrderOperationService.getPPGResult(new GetDbResultPPGCommand
            {
                Status = command.Status,
                Setting = setting,
                FinalAmount = dbResultCheckTransaction.Data.PaidAmount,
                CompanyId = dbResultCheckTransaction.Data.CompanyId,
                OrderId = dbResultCheckTransaction.Data.OrderId,
                TrackingCode = dbResultCheckTransaction.Data.TrackingCode,
                TransactionCode = dbResultCheckTransaction.Data.TransactionCode,
                PaidCode = dbResultCheckTransaction.Data.PaidCode
            });

        }
        /// <summary>
        /// شروع پرداخت جهت شارژ کیف پول - درگاه جیبیت
        /// </summary>
        /// <param name="command"></param>
        /// <param name="cancellationToken"></param>
        /// <returns></returns>
        /// <exception cref="LogicalException"></exception>
        public async Task<DBResult> startPayChargeWallet(StartPayChargeWalletCommand command, CancellationToken cancellationToken)
        {
            string token = _tokenUserService.getUserInfo().Token;

            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGJibit, cancellationToken);
            
            if (webServiceInfo == null)
                throw new LogicalException("درگاه پرداخت غیر فعال می باشد ، لطفا درگاه پرداخت دیگری را انتخاب کنید");
            
            var config = webServiceInfo.AdditionalConfig.JsonParse();

            if (setting.IsActiveOnlinePaySingle != true)
            {
                throw new LogicalException("روش پرداخت درگاه آنلاین غیر فعال می باشد ، لطفا روش پرداخت دیگری را انتخاب کنید");
            }

            
            var localIp = _httpContextAccessor.HttpContext?.Connection.LocalIpAddress;
            var isExternal = await IpCheckHelper.IsExternalServerAsync(setting.ExternalServerIp, new IpifyPublicIpProvider(), localIp);
            if (isExternal)
            {
                throw new LogicalException(MessageHelper.ExrenalServerErrorPayment);
            }

            await _IAutomaticOrderOperationService.CheckIpClientPortal(cancellationToken);

            var finalAmountToman = Convert.ToDecimal(command.PaidAmount);
            finalAmountToman.ValidatePaymentAmount(Convert.ToDecimal(setting.MaximumAmountOnlinePort), Convert.ToDecimal(setting.MinimumAmountOnlinePort));

            await _ITransActionContext.Check_TransAction_Wallet(new Check_TransAction_Wallet.Inputs
            {
                Token = token,
                PaidAmount = finalAmountToman,
                TransactionType = (int)TransactionType.RechargeWallet
            }, cancellationToken);


            string refrenceNumber = Guid.NewGuid().ToString();
            var jibitToken = await GetJibitTokenAsync(true, cancellationToken);

            var responsePurchasesJibitPPG = await _httpRequestJibitPPG.purchaseJibitPPG(new RequestPurchasesJibitPPG
            {
                amount = finalAmountToman.toRial(),
                callbackUrl = (string)config["callbackWallet"] + ("?returnUrl=" + command.CallbackUrl).Trim(),
                clientReferenceNumber = refrenceNumber,
                checkPayerNationalCode = false,
                switching = new RequestPurchasesJibitSwitchingPPG { autoSwitching = (string)config["smartSwitch"] == "1"?true:false, terminalIds = !string.IsNullOrWhiteSpace((string)config["terminal"]) ? config["terminal"].ToString().Split(",").ToList() : new List<string>() },
            }, jibitToken, webServiceInfo.BaseUrlService, cancellationToken);

            responsePurchasesJibitPPG.handlerRequestPPG(false);

            if (responsePurchasesJibitPPG.errors != null && responsePurchasesJibitPPG.errors.Count > 0 && responsePurchasesJibitPPG.errors[0].code == "token.verification_failed")
            {
                jibitToken = await GetJibitTokenAsync(false, cancellationToken);

                responsePurchasesJibitPPG = await _httpRequestJibitPPG.purchaseJibitPPG(new RequestPurchasesJibitPPG
                {
                    amount = finalAmountToman.toRial(),
                    callbackUrl = (string)config["callbackWallet"] + ("?returnUrl=" + command.CallbackUrl).Trim(),
                    clientReferenceNumber = refrenceNumber,
                    checkPayerNationalCode = false,
                    switching = new RequestPurchasesJibitSwitchingPPG { autoSwitching = (string)config["smartSwitch"] == "1" ? true : false, terminalIds = !string.IsNullOrWhiteSpace((string)config["terminal"]) ? config["terminal"].ToString().Split(",").ToList() : new List<string>() },
                }, jibitToken, webServiceInfo.BaseUrlService, cancellationToken);
            }

            responsePurchasesJibitPPG.handlerRequestPPG(true);

            await _ITransActionContext.Insert_Transaction_Wallet(new Insert_Transaction_Wallet.Inputs
            {
                Token = token,
                PaidAmount = finalAmountToman,
                PaidCode = responsePurchasesJibitPPG.purchaseId.ToString(),
                TrackingCode = refrenceNumber,
                OnlinePortal = (int)OnlinePortal.Jibit,
                TransactionType = (int)TransactionType.RechargeWallet
            }, cancellationToken);
            return new DBResult(new { redirectUrl = responsePurchasesJibitPPG.pspSwitchingUrl }, 1);
        }

        public async Task<DBResult> verifyChargeWallet(VerifyPayChargeCommand command, CancellationToken cancellationToken)
        {
            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGJibit, cancellationToken);

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var dbResultCheckTransaction = await _ITransActionContext.Check_Duplicate_Transaction_Wallet(new Check_Duplicate_Transaction_Wallet.Inputs
            {
                PaidCode = command.PurchaseId,
                Status = command.Status
            }, cancellationToken);

            if (dbResultCheckTransaction != null && dbResultCheckTransaction.Data != null && dbResultCheckTransaction.Data.IsDuplicate == true)
            {

                return _IAutomaticOrderOperationService.getPPGResult(new GetDbResultPPGCommand
                {
                    Status = "SUCCESSFUL",
                    Setting = setting,
                    FinalAmount = dbResultCheckTransaction.Data.PaidAmount,
                    CompanyId = dbResultCheckTransaction.Data.CompanyId,
                    TrackingCode = dbResultCheckTransaction.Data.TrackingCode,
                    TransactionCode = dbResultCheckTransaction.Data.TransactionCode,
                    PaidCode = dbResultCheckTransaction.Data.PaidCode,
                    WalletId = dbResultCheckTransaction.Data.WalletId,
                });

            }

            if (command.Status == "SUCCESSFUL")
            {
                var jibitToken = await GetJibitTokenAsync(true, cancellationToken);
                var responsePurchasesVerifyJibitPPG = await _httpRequestJibitPPG.verifyJibitPPG(command.PurchaseId
                     , jibitToken
                     , webServiceInfo.BaseUrlService, cancellationToken);

                if (responsePurchasesVerifyJibitPPG == null)
                {
                    throw new ExternalServiceException("خطا در دریافت نتیجه ی درگاه پرداخت ، در صورت کسر وجه مبلغ پرداخت شده طی 24 ساعت به حساب شما برگشت داده خواهد شد");
                }

                if (responsePurchasesVerifyJibitPPG.errors != null && responsePurchasesVerifyJibitPPG.errors.Count > 0 && responsePurchasesVerifyJibitPPG.errors[0].code == "purchase.not_found")
                {
                    throw new ExternalServiceException("کد پرداخت ارسال شده اشتباه است ، در صورت کسر وجه مبلغ پرداخت شده طی 24 ساعت به حساب شما برگشت داده خواهد شد");
                }


                if (responsePurchasesVerifyJibitPPG.errors != null && responsePurchasesVerifyJibitPPG.errors.Count > 0 && responsePurchasesVerifyJibitPPG.errors[0].code == "token.verification_failed")
                {
                    jibitToken = await GetJibitTokenAsync(false, cancellationToken);

                    responsePurchasesVerifyJibitPPG = await _httpRequestJibitPPG.verifyJibitPPG(command.PurchaseId
                        , jibitToken, webServiceInfo.BaseUrlService, cancellationToken);

                    if (responsePurchasesVerifyJibitPPG == null)
                    {
                        throw new ExternalServiceException("خطا در دریافت نتیجه ی درگاه پرداخت ، در صورت کسر وجه مبلغ پرداخت شده طی 24 ساعت به حساب شما برگشت داده خواهد شد");
                    }

                    if (responsePurchasesVerifyJibitPPG.errors != null && responsePurchasesVerifyJibitPPG.errors.Count > 0 && responsePurchasesVerifyJibitPPG.errors[0].code == "purchase.not_found")
                    {
                        throw new ExternalServiceException("کد پرداخت ارسال شده اشتباه است ، در صورت کسر وجه مبلغ پرداخت شده طی 24 ساعت به حساب شما برگشت داده خواهد شد");
                    }


                }
                if (responsePurchasesVerifyJibitPPG.errors != null && responsePurchasesVerifyJibitPPG.errors.Count > 0)
                {

                    throw new ExternalServiceException("خطا در ارسال درخواست به درگاه آنلاین ، در صورت کسر وجه مبلغ پرداخت شده طی 24 ساعت به حساب شما برگشت داده خواهد شد");
                }

                if (responsePurchasesVerifyJibitPPG.status == "SUCCESSFUL")
                {
                    var dbResultUpdateTransaction = await _ITransActionContext.Update_Transaction_Wallet(new Update_Transaction_Wallet.Inputs
                    {
                        TransactionCode = command.TransactionCode,
                        PaidAmount = Convert.ToDecimal(command.PaidAmount) / 10,
                        PaidCode = command.PurchaseId,
                        Status = command.Status,
                        TrackingCode = command.TrackingCode
                    }, cancellationToken);


                    return await _IAutomaticOrderOperationService.doCompeleteWalletCharge(
                    new DoCompeleteWalletCharge
                    {
                        PaidAmount = dbResultUpdateTransaction.Data.PaidAmount,
                        Email = dbResultUpdateTransaction.Data.Email,
                        Mobile = dbResultUpdateTransaction.Data.Mobile,
                        Status = responsePurchasesVerifyJibitPPG.status,
                        CompanyId = dbResultUpdateTransaction.Data.CompanyId,
                        Setting = setting,
                        PaidCode = dbResultUpdateTransaction.Data.PaidCode,
                        TransactionCode = dbResultUpdateTransaction.Data.TransactionCode,
                        TrackingCode = dbResultUpdateTransaction.Data.TrackingCode,
                        WalletId = dbResultUpdateTransaction.Data.WalletId,
                    }, cancellationToken);
                }

                return _IAutomaticOrderOperationService.getPPGResult(new GetDbResultPPGCommand
                {
                    Status = responsePurchasesVerifyJibitPPG.status,
                    Setting = setting,
                    FinalAmount = dbResultCheckTransaction?.Data?.PaidAmount,
                    CompanyId = dbResultCheckTransaction.Data.CompanyId,
                    TrackingCode = dbResultCheckTransaction.Data.TrackingCode,
                    TransactionCode = dbResultCheckTransaction.Data.TransactionCode,
                    PaidCode = dbResultCheckTransaction.Data.PaidCode,
                    WalletId = dbResultCheckTransaction.Data.WalletId,
                });
            }

            return _IAutomaticOrderOperationService.getPPGResult(new GetDbResultPPGCommand
            {
                Status = command.Status,
                Setting = setting,
                FinalAmount = dbResultCheckTransaction?.Data?.PaidAmount,
                CompanyId = dbResultCheckTransaction.Data.CompanyId,
                TrackingCode = dbResultCheckTransaction.Data.TrackingCode,
                TransactionCode = dbResultCheckTransaction.Data.TransactionCode,
                PaidCode = dbResultCheckTransaction.Data.PaidCode,
                WalletId = dbResultCheckTransaction.Data.WalletId,
            });
        }

        public async Task<DBResult<List<ResponsePurchaseFilterItemJibitPPG>>> GetPayList(GetPayJibitCommand command, CancellationToken cancellationToken, bool checkJobToken = true)
        {
            if (checkJobToken)
            {
                var dbSetting = await _ISettingContext.Get_Setting_Job(new Get_Setting_Job.Inputs { }, cancellationToken);
                if (dbSetting.Data.JobToken != command.JobToken)
                {
                    throw new LogicalException("کلید اتصال نامعتبر می باشد");
                }
            }
            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGJibit, cancellationToken);

            
            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var dbResult = await _IQueueContext.Get_Last_Token(new Get_Last_Token.Inputs { TokenType = (int)TokenType.JibitPPG }, cancellationToken);

            if (string.IsNullOrWhiteSpace(dbResult.Data.JibitTokenPPG))
            {
                var responseJibit = await _httpRequestJibitPPG.getTokenJibitPPG(webServiceInfo.RealApiKey, webServiceInfo.RealPassword,
                    webServiceInfo.BaseUrlService, cancellationToken);
                if (responseJibit == null || string.IsNullOrWhiteSpace(responseJibit.accessToken))
                {
                    throw new ExternalServiceException("خطا در دریافت امضای درگاه پرداخت ، در صورت تکرار خطا مراتب را پشتیبانی برنامه اطلاع دهید");
                }

                dbResult.Data.JibitTokenPPG = responseJibit.accessToken;
                await _IQueueContext.Insert_Token(new Insert_Token.Inputs { JibitTokenPPG = responseJibit.accessToken }, cancellationToken);
            }


            var responsePurchaseFilterJibitPPG = new ResponsePurchaseFilterJibitPPG();
            responsePurchaseFilterJibitPPG = await _httpRequestJibitPPG.getPurchaseJibitPPG(new RequestPurchaseFilterJibitPPG
            {
                purchaseId = command.PurchaseId,
                clientReferenceNumber = command.ClientReferenceNumber,
                from = command.From,
                page = command.Page,
                pspReferenceNumber = command.PspReferenceNumber,
                pspRrn = command.PspRrn,
                pspTraceNumber = command.PspTraceNumber,
                size = command.Size,
                status = command.Status,
                to = command.To,
                userIdentifier = command.UserIdentifier
            }
            , dbResult.Data.JibitTokenPPG, webServiceInfo.BaseUrlService, cancellationToken);

            if (responsePurchaseFilterJibitPPG != null && responsePurchaseFilterJibitPPG.errors != null && responsePurchaseFilterJibitPPG.errors.Count > 0 && responsePurchaseFilterJibitPPG.errors[0].code == "token.verification_failed")
            {
                var responseJibit = await _httpRequestJibitPPG.getTokenJibitPPG(webServiceInfo.RealApiKey, webServiceInfo.RealPassword
                    , webServiceInfo.BaseUrlService, cancellationToken);
                if (responseJibit == null || string.IsNullOrWhiteSpace(responseJibit.accessToken))
                {
                    throw new ExternalServiceException("خطا در دریافت امضای درگاه پرداخت ، در صورت تکرار خطا مراتب را پشتیبانی برنامه اطلاع دهید");
                }

                dbResult.Data.JibitTokenPPG = responseJibit.accessToken;
                await _IQueueContext.Insert_Token(new Insert_Token.Inputs { JibitTokenPPG = responseJibit.accessToken }, cancellationToken);
                responsePurchaseFilterJibitPPG = await _httpRequestJibitPPG.getPurchaseJibitPPG(new RequestPurchaseFilterJibitPPG
                {
                    purchaseId = command.PurchaseId,
                    clientReferenceNumber = command.ClientReferenceNumber,
                    from = command.From,
                    page = command.Page,
                    pspReferenceNumber = command.PspReferenceNumber,
                    pspRrn = command.PspRrn,
                    pspTraceNumber = command.PspTraceNumber,
                    size = command.Size,
                    status = command.Status,
                    to = command.To,
                    userIdentifier = command.UserIdentifier
                }
, dbResult.Data.JibitTokenPPG, webServiceInfo.BaseUrlService, cancellationToken);
            }

            if (responsePurchaseFilterJibitPPG != null && responsePurchaseFilterJibitPPG.errors != null && responsePurchaseFilterJibitPPG.errors.Count > 0)
                throw new ExternalServiceException($"{responsePurchaseFilterJibitPPG.errors[0]}");

            if (responsePurchaseFilterJibitPPG != null && responsePurchaseFilterJibitPPG.elements != null && responsePurchaseFilterJibitPPG.elements.Count > 0)
            {
                return new DBResult<List<ResponsePurchaseFilterItemJibitPPG>> { Data = responsePurchaseFilterJibitPPG.elements };
            }

            throw new ExternalServiceException("تراکنش یافت نشد");


        }

        public async Task<DBResult<ResponsePurchaseFilterItemJibitPPG>> getPay(GetPayJibitCommand command, CancellationToken cancellationToken, bool checkJobToken = true)
        {
            if (checkJobToken)
            {
                var dbSetting = await _ISettingContext.Get_Setting_Job(new Get_Setting_Job.Inputs { }, cancellationToken);
                if (dbSetting.Data.JobToken != command.JobToken)
                {
                    throw new LogicalException("کلید اتصال نامعتبر می باشد");
                }
            }
            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGJibit, cancellationToken);

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var dbResult = await _IQueueContext.Get_Last_Token(new Get_Last_Token.Inputs { TokenType = (int)TokenType.JibitPPG }, cancellationToken);

            if (string.IsNullOrWhiteSpace(dbResult.Data.JibitTokenPPG))
            {
                var responseJibit = await _httpRequestJibitPPG.getTokenJibitPPG(webServiceInfo.RealApiKey, webServiceInfo.RealPassword,
                    webServiceInfo.BaseUrlService, cancellationToken);
                if (responseJibit == null || string.IsNullOrWhiteSpace(responseJibit.accessToken))
                {
                    throw new ExternalServiceException("خطا در دریافت امضای درگاه پرداخت ، در صورت تکرار خطا مراتب را پشتیبانی برنامه اطلاع دهید");
                }

                dbResult.Data.JibitTokenPPG = responseJibit.accessToken;
                await _IQueueContext.Insert_Token(new Insert_Token.Inputs { JibitTokenPPG = responseJibit.accessToken }, cancellationToken);
            }


            var responsePurchaseFilterJibitPPG = new ResponsePurchaseFilterJibitPPG();
            responsePurchaseFilterJibitPPG = await _httpRequestJibitPPG.getPurchaseJibitPPG(new RequestPurchaseFilterJibitPPG { purchaseId = command.PurchaseId }
            , dbResult.Data.JibitTokenPPG, webServiceInfo.BaseUrlService, cancellationToken);

            if (responsePurchaseFilterJibitPPG != null && responsePurchaseFilterJibitPPG.errors != null && responsePurchaseFilterJibitPPG.errors.Count > 0 && responsePurchaseFilterJibitPPG.errors[0].code == "token.verification_failed")
            {
                var responseJibit = await _httpRequestJibitPPG.getTokenJibitPPG(webServiceInfo.RealApiKey, webServiceInfo.RealPassword
                    , webServiceInfo.BaseUrlService, cancellationToken);
                if (responseJibit == null || string.IsNullOrWhiteSpace(responseJibit.accessToken))
                {
                    throw new ExternalServiceException("خطا در دریافت امضای درگاه پرداخت ، در صورت تکرار خطا مراتب را پشتیبانی برنامه اطلاع دهید");
                }

                dbResult.Data.JibitTokenPPG = responseJibit.accessToken;
                await _IQueueContext.Insert_Token(new Insert_Token.Inputs { JibitTokenPPG = responseJibit.accessToken }, cancellationToken);
                responsePurchaseFilterJibitPPG = await _httpRequestJibitPPG.getPurchaseJibitPPG(new RequestPurchaseFilterJibitPPG
                {
                    purchaseId = command.PurchaseId,
                    clientReferenceNumber = command.ClientReferenceNumber,
                    from = command.From,
                    page = command.Page,
                    pspReferenceNumber = command.PspReferenceNumber,
                    pspRrn = command.PspRrn,
                    pspTraceNumber = command.PspTraceNumber,
                    size = command.Size,
                    status = command.Status,
                    to = command.To,
                    userIdentifier = command.UserIdentifier
                }
, dbResult.Data.JibitTokenPPG, webServiceInfo.BaseUrlService, cancellationToken);
            }

            if (responsePurchaseFilterJibitPPG != null && responsePurchaseFilterJibitPPG.errors != null && responsePurchaseFilterJibitPPG.errors.Count > 0)
                throw new ExternalServiceException($"{responsePurchaseFilterJibitPPG.errors[0]}");

            if (responsePurchaseFilterJibitPPG != null && responsePurchaseFilterJibitPPG.elements != null && responsePurchaseFilterJibitPPG.elements.Count > 0)
            {
                return new DBResult<ResponsePurchaseFilterItemJibitPPG> { Data = responsePurchaseFilterJibitPPG.elements.FirstOrDefault() };
            }

            throw new ExternalServiceException("تراکنش یافت نشد");


        }


    }
}
