﻿using Farakonesh.Commands.Services;
using Farakonesh.Commands.Services.Order;
using Farakonesh.Logic.CommonOperations;
using Farakonesh.Logic.ICommonOperations;
using Farakonesh.Logic.IDatabase.idbo;
using Farakonesh.Logic.IDatabase.IOrder;
using Farakonesh.Logic.IDatabase.IUser;
using Farakonesh.Logic.ISecurity;
using Farakonesh.Logic.IServices.IApp.IOrder;
using Farakonesh.Logic.IServices.IApp.IPPG;
using Farakonesh.Logic.IServices.ICache;
using Farakonesh.Logic.IServices.IRestRequest;
using Farakonesh.Logic.Services.Cache;
using Farakonesh.Logic.Services.RestRequest;
using Farakonesh.Models.API.Shepa;
using Farakonesh.Models.API.ZarinLink;
using Farakonesh.Models.API.Zarinpal;
using Farakonesh.Models.Database;
using Farakonesh.Models.Database.StoredProcedures.App.dbo.setting;
using Farakonesh.Models.Database.StoredProcedures.App.Order;
using Farakonesh.Models.Database.StoredProcedures.App.Order.Orders;
using Farakonesh.Models.Database.StoredProcedures.App.Order.PaymentLink;
using Farakonesh.Models.Database.StoredProcedures.App.Order.Transaction;
using Farakonesh.Shared.Enums;
using Farakonesh.Shared.Exceptions;
using Farakonesh.Shared.Helpers;
using Microsoft.AspNetCore.Http;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace Farakonesh.Logic.Services.App.PPG
{
    public class ZarinLinkService : IZarinLinkService
    {
        private readonly IOrderContext _IOrderContext;
        private readonly IPaymentLinkContext _paymentLinkContext;
        private readonly ISettingContext _ISettingContext;
        private readonly ITransActionContext _ITransActionContext;
        private readonly IHttpRequestZarinLink _httpRequestZarinLink;
        private readonly ITokenUserService _tokenUserService;
        private readonly IAutomaticOrderOperationService _IAutomaticOrderOperationService;
        private readonly IRedisContextService _redisContextService;
        private readonly IHttpContextAccessor _httpContextAccessor;
        public ZarinLinkService(IAutomaticOrderOperationService IAutomaticOrderOperationService
            , ISettingContext iSettingContext, IOrderContext iOrderContext
            , ITransActionContext iTransActionContext, IHttpRequestZarinLink httpRequestZarinLink,
            ITokenUserService tokenUserService, IPaymentLinkContext paymentLinkContext,
            IRedisContextService redisContextService, IHttpContextAccessor httpContextAccessor)
        {
            _tokenUserService = tokenUserService;
            _httpRequestZarinLink = httpRequestZarinLink;
            _IAutomaticOrderOperationService = IAutomaticOrderOperationService;
            _ISettingContext = iSettingContext;
            _IOrderContext = iOrderContext;
            _ITransActionContext = iTransActionContext;
            _paymentLinkContext = paymentLinkContext;
            _redisContextService = redisContextService;
            _httpContextAccessor = httpContextAccessor;
        }

        public async Task<DBResult> startPay(StartPayOrderCommand command, CancellationToken cancellationToken)
        {
            string token = _tokenUserService.getUserInfo().Token;
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGZarinlink, cancellationToken);

            if (webServiceInfo == null)
                throw new LogicalException("درگاه پرداخت غیر فعال می باشد ، لطفا درگاه پرداخت دیگری را انتخاب کنید");

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var dbResultOrder = await _IOrderContext.GetOrderByIdWithoutToken(new GetOrderByIdWithoutToken.Inputs
            {
                OrderId = command.OrderId,
                Code = command.Code
            }, cancellationToken);

            if (dbResultOrder == null || dbResultOrder.Data == null)
            {
                throw new LogicalException(MessageHelper.NotFoundOrder);
            }

            var allowedAmount = await _IAutomaticOrderOperationService.CheckScenario(new CheckScenarioCommand
            {
                OrderType = dbResultOrder.Data.OrderType,
                PaidAmount = dbResultOrder.Data.PaidAmount,
                FinalAmount = dbResultOrder.Data.FinalAmount,
                FinalWageAmount = dbResultOrder.Data.FinalWageAmount,
                PaymentMethod = PayMentMethod.OnlinePayment,
                DiscountedFinalAmount = dbResultOrder.Data.DiscountedFinalAmount,
                ModuleType = dbResultOrder.Data.SaveLocation
            }, cancellationToken);

            var responseFreeOrder = await _IAutomaticOrderOperationService.doCompeleteFreeOrder(new DoCompeleteFreeOrderCommand
            {
                Code = command.Code,
                OrderId = command.OrderId,
                model = dbResultOrder.Data,
                CheckToken = true
            }, cancellationToken);
            if (responseFreeOrder != null)
                return responseFreeOrder;

            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var localIp = _httpContextAccessor.HttpContext?.Connection.LocalIpAddress;
            var isExternal = await IpCheckHelper.IsExternalServerAsync(setting.ExternalServerIp, new IpifyPublicIpProvider(), localIp);
            if (isExternal)
            {
                throw new LogicalException(MessageHelper.ExrenalServerErrorPayment);
            }
            var settingOrder = await _redisContextService.getSettingOrderAsync(dbResultOrder.Data.OrderType, token, cancellationToken);

 

            var dbResultBankAccount = await _IAutomaticOrderOperationService.checkBankAccountClientPortal(
            new CheckBankAccountClientPortalCommand
            {
                BankAccountId = command.BankAccountId,
                CheckBankAccount = settingOrder.CheckBankAccount,
                Token = token
            }, cancellationToken);

            await _IAutomaticOrderOperationService.CheckIpClientPortal(cancellationToken);
            var maximumAmountOnlinePort = Convert.ToDecimal(settingOrder.MaximumAmountOnlinePort);
            var minimumAmountOnlinePort = Convert.ToDecimal(settingOrder.MinimumAmountOnlinePort);
            var finalAmountToman = (allowedAmount != null && allowedAmount > 0) ? allowedAmount : dbResultOrder.Data.Discount != null && dbResultOrder.Data.Discount > 0 ? dbResultOrder.Data.DiscountedFinalAmount : dbResultOrder.Data.FinalAmount;
            finalAmountToman = finalAmountToman.Value.ValidatePaymentAmountMultiPayment(maximumAmountOnlinePort,
minimumAmountOnlinePort, dbResultOrder.Data.PaidAmount, settingOrder.IsActiveOnlinePaySingle,
settingOrder.IsActiveOnlinePayMultiple);

            var zarinLink = await getActiveZarinLink(finalAmountToman, cancellationToken);

            var responsePurchasesZarinpalPPG = await _httpRequestZarinLink.generateZarinLinkV2(new RequestGenerateZarinLinkV2
            {
                amount = finalAmountToman.toRial(),
                terminal_id = zarinLink.Terminal,
                title = "سفارش شماره " + dbResultOrder.Data.OrderNumber,
                successful_redirect_url = (string)config["callback"],
                failed_redirect_url = (string)config["callbackFailed"],
                description = "سفارش شماره " + dbResultOrder.Data.OrderNumber
            }, webServiceInfo.BaseUrlService, zarinLink.AccessToken, cancellationToken);


            if (responsePurchasesZarinpalPPG == null)
                throw new ExternalServiceException("خطا در اتصال به درگاه پرداخت ، در صورت تکرار خطا با کارشناسان ما ارتباط برقرار کنید");

            await _ITransActionContext.Insert_Transaction(new Insert_Transaction.Inputs
            {
                Token = token,
                OrderId = command.OrderId,
                PaidAmount = finalAmountToman,
                PaidCode = responsePurchasesZarinpalPPG.data.NewZarinLinkAdd.id,
                TrackingCode = zarinLink.Terminal,
                MaximummPrice = maximumAmountOnlinePort,
                OnlinePortal = (int)OnlinePortal.ZarinLink,
                Code = command.Code
            }, cancellationToken);

            return new DBResult(new { paidCode = responsePurchasesZarinpalPPG.data.NewZarinLinkAdd.id, redirectUrl = (string)config["redirectUrl"] + responsePurchasesZarinpalPPG.data.NewZarinLinkAdd.id, IsFree = false }, 1);
        }

        public async Task<DBResult> startPayPublic(StartPayPublicOrderCommand command, CancellationToken cancellationToken)
        {

            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGZarinlink, cancellationToken);

            if (webServiceInfo == null)
                throw new LogicalException("درگاه پرداخت غیر فعال می باشد ، لطفا درگاه پرداخت دیگری را انتخاب کنید");

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var dbResultOrder = await _IOrderContext.GetOrderByIdWithoutToken(new GetOrderByIdWithoutToken.Inputs
            {
                OrderNumber = command.OrderNumber,
                Code = command.Code
            }, cancellationToken);
            if (dbResultOrder == null || dbResultOrder.Data == null)
            {
                throw new LogicalException(MessageHelper.NotFoundOrder);
            }
            command.OrderId = dbResultOrder.Data.OrderId;

            var settingOrder = await _redisContextService.getSettingOrderWithoutTokenAsync(dbResultOrder.Data.OrderType
                , command.OrderId, cancellationToken);

            if (settingOrder.IsActiveZarinLink == null || settingOrder.IsActiveZarinLink == false)
                throw new LogicalException("درگاه پرداخت غیر فعال می باشد ، لطفا درگاه پرداخت دیگری را انتخاب کنید");


            var allowedAmount = await _IAutomaticOrderOperationService.CheckScenario(new CheckScenarioCommand
            {
                OrderType = dbResultOrder.Data.OrderType,
                PaidAmount = dbResultOrder.Data.PaidAmount,
                FinalAmount = dbResultOrder.Data.FinalAmount,
                FinalWageAmount = dbResultOrder.Data.FinalWageAmount,
                PaymentMethod = PayMentMethod.OnlinePayment,
                DiscountedFinalAmount = dbResultOrder.Data.DiscountedFinalAmount,
                ModuleType = dbResultOrder.Data.SaveLocation
            }, cancellationToken);

            var responseFreeOrder = await _IAutomaticOrderOperationService.doCompeleteFreeOrder(new DoCompeleteFreeOrderCommand
            {
                Code = command.Code,
                OrderId = command.OrderId,
                model = dbResultOrder.Data,
                CheckToken = false
            }, cancellationToken);
            if (responseFreeOrder != null)
                return responseFreeOrder;

            var dbResultBankAccount = await _IAutomaticOrderOperationService.checkBankAccountClientPortalWithoutToken(
            new CheckBankAccountClientPortalCommand
            {
                BankAccountId = command.BankAccountId,
                CheckBankAccount = settingOrder.CheckBankAccount
            }, cancellationToken);

            await _IAutomaticOrderOperationService.CheckIpClientPortal(cancellationToken);


            var maximumAmountOnlinePort = Convert.ToDecimal(settingOrder.MaximumAmountOnlinePort);
            var minimumAmountOnlinePort = Convert.ToDecimal(settingOrder.MinimumAmountOnlinePort);
            var finalAmountToman = (allowedAmount != null && allowedAmount > 0) ? allowedAmount : dbResultOrder.Data.Discount != null && dbResultOrder.Data.Discount > 0 ? dbResultOrder.Data.DiscountedFinalAmount : dbResultOrder.Data.FinalAmount;
            finalAmountToman = finalAmountToman.Value.ValidatePaymentAmountMultiPayment(maximumAmountOnlinePort,
   minimumAmountOnlinePort, dbResultOrder.Data.PaidAmount, settingOrder.IsActiveOnlinePaySingle,
   settingOrder.IsActiveOnlinePayMultiple);

            var zarinLink = await getActiveZarinLink(finalAmountToman, cancellationToken);

            var responsePurchasesZarinpalPPG = await _httpRequestZarinLink.generateZarinLinkV2(new RequestGenerateZarinLinkV2
            {
                amount = finalAmountToman.toRial(),
                successful_redirect_url = (string)config["callback"],
                failed_redirect_url = (string)config["callbackFailed"],
                description = "سفارش شماره " + dbResultOrder.Data.OrderNumber,
                title = "سفارش شماره " + dbResultOrder.Data.OrderNumber,
                terminal_id = zarinLink.Terminal
            }, webServiceInfo.BaseUrlService, zarinLink.AccessToken, cancellationToken);


            if (responsePurchasesZarinpalPPG == null)
                throw new ExternalServiceException("خطا در اتصال به درگاه پرداخت ، در صورت تکرار خطا با کارشناسان ما ارتباط برقرار کنید");

            await _ITransActionContext.Insert_Transaction_WithoutToken(new Insert_Transaction_WithoutToken.Inputs
            {
                UserId = dbResultOrder.Data.UserId,
                OrderId = command.OrderId,
                PaidAmount = finalAmountToman,
                PaidCode = responsePurchasesZarinpalPPG.data.NewZarinLinkAdd.id,
                TrackingCode = zarinLink.Terminal,
                MaximummPrice = maximumAmountOnlinePort,
                OnlinePortal = (int)OnlinePortal.ZarinLink,
                Code = command.Code
            }, cancellationToken);
            return new DBResult(new { paidCode = responsePurchasesZarinpalPPG.data.NewZarinLinkAdd.id, redirectUrl = (string)config["redirectUrl"] + responsePurchasesZarinpalPPG.data.NewZarinLinkAdd.id, IsFree = false }, 1);
        }

        public async Task<DBResult> startPayChargeWallet(StartPayChargeWalletCommand command, CancellationToken cancellationToken)
        {
            string token = _tokenUserService.getUserInfo().Token;

            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGZarinlink, cancellationToken);

            if (webServiceInfo == null)
                throw new LogicalException("درگاه پرداخت غیر فعال می باشد ، لطفا درگاه پرداخت دیگری را انتخاب کنید");

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            if ((setting.IsActiveOnlinePaySingle == null || setting.IsActiveOnlinePaySingle == false))
            {
                throw new LogicalException("روش پرداخت درگاه آنلاین غیر فعال می باشد ، لطفا روش پرداخت دیگری را انتخاب کنید");
            }

            var localIp = _httpContextAccessor.HttpContext?.Connection.LocalIpAddress;
            var isExternal = await IpCheckHelper.IsExternalServerAsync(setting.ExternalServerIp, new IpifyPublicIpProvider(), localIp);
            if (isExternal)
            {
                throw new LogicalException(MessageHelper.ExrenalServerErrorPayment);
            }

            await _IAutomaticOrderOperationService.CheckIpClientPortal(cancellationToken);
            var maximumAmountOnlinePort = Convert.ToDecimal(setting.MaximumAmountOnlinePort);
            var minimumAmountOnlinePort = Convert.ToDecimal(setting.MinimumAmountOnlinePort);

            var finalAmountToman = Convert.ToDecimal(command.PaidAmount);
            finalAmountToman.ValidatePaymentAmount(maximumAmountOnlinePort, minimumAmountOnlinePort);
            var responseCheckTransaction = await _ITransActionContext.Check_TransAction_Wallet(new Check_TransAction_Wallet.Inputs
            {
                Token = token,
                PaidAmount = finalAmountToman,
                TransactionType = (int)TransactionType.RechargeWallet
            }, cancellationToken);
            var zarinLink = await getActiveZarinLink(finalAmountToman, cancellationToken);

            var responsePurchasesZarinpalPPG = await _httpRequestZarinLink.generateZarinLinkV2(new RequestGenerateZarinLinkV2
            {
                amount = finalAmountToman.toRial(),
                successful_redirect_url = (string)config["callbackWallet"] + "-wallet",
                failed_redirect_url = (string)config["callbackFailed"],
                description = "شارژ کیف پول آقا/خانم " + responseCheckTransaction.Data.Name + " " + responseCheckTransaction.Data.Family + " با شماره همراه: " + responseCheckTransaction.Data.Mobile + " و ایمیل: " + responseCheckTransaction.Data.Email,
                title = "شارژ کیف پول آقا/خانم " + responseCheckTransaction.Data.Name + " " + responseCheckTransaction.Data.Family + " با شماره همراه: " + responseCheckTransaction.Data.Mobile + " و ایمیل: " + responseCheckTransaction.Data.Email,
                terminal_id = zarinLink.Terminal
            }, webServiceInfo.BaseUrlService, zarinLink.AccessToken, cancellationToken);



            if (responsePurchasesZarinpalPPG == null)
                throw new ExternalServiceException("خطا در اتصال به درگاه پرداخت ، در صورت تکرار خطا با کارشناسان ما ارتباط برقرار کنید");


            await _ITransActionContext.Insert_Transaction_Wallet(new Insert_Transaction_Wallet.Inputs
            {
                Token = token,
                PaidAmount = finalAmountToman,
                PaidCode = responsePurchasesZarinpalPPG.data.NewZarinLinkAdd.id,
                TrackingCode = zarinLink.Terminal,
                OnlinePortal = (int)OnlinePortal.ZarinLink,
                TransactionType = (int)TransactionType.RechargeWallet
            }, cancellationToken);

            return new DBResult(new { paidCode = responsePurchasesZarinpalPPG.data.NewZarinLinkAdd.id, redirectUrl = (string)config["redirectUrl"] + responsePurchasesZarinpalPPG.data.NewZarinLinkAdd.id }, 1);
        }


        public async Task<DBResult> verifyPay(VerifyPayOrderCommand command, CancellationToken cancellationToken)
        {
            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGZarinlink, cancellationToken);

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var dbResultUpdateTransaction = new DBResult<Update_Transaction.Outputs>();
            var dbResultGetTransactionOrder = await _ITransActionContext.Get_Order_Transaction_WithouToken(new Get_Order_Transaction_WithouToken.Inputs
            {
                PaidCode = command.PurchaseId
            }, cancellationToken);
            decimal paidAmountOrder = dbResultGetTransactionOrder.Data.PaidAmount.Value;

            command.PaidAmount = Math.Round(dbResultGetTransactionOrder.Data.PaidAmount.Value).ToString();

            decimal price = command.PaidAmount.cleanPrice();

            var dbResultCheckTransaction = await _ITransActionContext.Check_Duplicate_Transaction(new Check_Duplicate_Transaction.Inputs
            {
                PaidCode = command.PurchaseId,
                Status = command.Status
            }, cancellationToken);

            if (dbResultCheckTransaction != null && dbResultCheckTransaction.Data != null && dbResultCheckTransaction.Data.IsDuplicate == true)
            {
                return _IAutomaticOrderOperationService.getPPGResult(new GetDbResultPPGCommand
                {
                    Status = "OK",
                    Setting = setting,
                    CompanyId = dbResultCheckTransaction.Data.CompanyId,
                    TrackingCode = dbResultCheckTransaction.Data.TrackingCode,
                    TransactionCode = dbResultCheckTransaction.Data.TransactionCode,
                    PaidCode = dbResultCheckTransaction.Data.PaidCode,
                    PaidDate = dbResultCheckTransaction.Data.PaidDate,
                    FinalAmount = dbResultCheckTransaction.Data.PaidAmount
                });
            }

            if (command.Status == "OK")
            {
                var responsePaymentLink = await _paymentLinkContext.GetPaymentLink_WithoutToken(
                    new GetPaymentLink_WithoutToken.Inputs { ExternalId = command.PurchaseId }, cancellationToken);
                var paymentLink = responsePaymentLink.Data.First();
                var responsePurchasesZarinpalPPG = await _httpRequestZarinLink.getTransactionListByZarinLinkId(new RequestGetTransactionZarinLink
                {
                    terminal_id = paymentLink.Terminal
                ,
                    relation_id = command.PurchaseId
                }
                , webServiceInfo.BaseUrlService, paymentLink.AccessToken, cancellationToken);

                var maximumAmountOnlinePort = Convert.ToDecimal(setting.MaximumAmountOnlinePort);

                if (responsePurchasesZarinpalPPG == null || responsePurchasesZarinpalPPG.data == null
                    || responsePurchasesZarinpalPPG.data.Session == null
                    || !responsePurchasesZarinpalPPG.data.Session.Any(q => q.status.Equals("PAID")))
                {
                    var dbResultUpdateStatus = await _IOrderContext.Update_Order_Status_WithouToken(new Update_Order_Status_WithouToken.Inputs
                    {
                        OrderId = dbResultGetTransactionOrder.Data.OrderId,
                        Status = (int)OrderStatus.NotConfirmed
                    }, cancellationToken);
                    throw new ExternalServiceException("خطا در پرداخت سفارش ، در صورت کسر وجه مبلغ طی 24 ساعت به حساب شما برگشت داده خواهد شد");
                }


                var responsePurchasesZarinLinkById = await _httpRequestZarinLink.getZarinLinkById(new RequestGetZarinLink
                {
                    terminal_id = paymentLink.Terminal,
                    id = command.PurchaseId
                }

, webServiceInfo.BaseUrlService, paymentLink.AccessToken, cancellationToken);
                if (responsePurchasesZarinLinkById == null || responsePurchasesZarinLinkById.data == null ||
                  responsePurchasesZarinLinkById.data.NewZarinLinks == null || responsePurchasesZarinLinkById.data.NewZarinLinks.Count == 0 || responsePurchasesZarinLinkById.data.NewZarinLinks.FirstOrDefault().deleted_at != null)
                {
                    var dbResultUpdateStatus = await _IOrderContext.Update_Order_Status_WithouToken(new Update_Order_Status_WithouToken.Inputs
                    {
                        OrderId = dbResultGetTransactionOrder.Data.OrderId,
                        Status = (int)OrderStatus.NotConfirmed
                    }, cancellationToken);
                    throw new ExternalServiceException("خطا در شارژ کیف پول ، در صورت کسر وجه مبلغ طی 24 ساعت به حساب شما برگشت داده خواهد شد");
                }

                var trackingCode = responsePurchasesZarinpalPPG?.data?.Session?.FirstOrDefault(q => q.status.Equals("PAID"))?.id;
                var responsePurchasesZarinLinkTransactionById = await _httpRequestZarinLink.getTransactionByIdByZarinLinkId(new RequestGetTransactionZarinLink
                {
                    terminal_id = paymentLink.Terminal,
                    relation_id = trackingCode
                }

                , webServiceInfo.BaseUrlService, paymentLink.AccessToken, cancellationToken);

                if (responsePurchasesZarinLinkTransactionById == null || responsePurchasesZarinLinkTransactionById.data == null ||
                responsePurchasesZarinLinkTransactionById.data.Session == null || responsePurchasesZarinLinkTransactionById.data.Session.Count == 0)
                {
                    var dbResultUpdateStatus = await _IOrderContext.Update_Order_Status_WithouToken(new Update_Order_Status_WithouToken.Inputs
                    {
                        OrderId = dbResultGetTransactionOrder.Data.OrderId,
                        Status = (int)OrderStatus.NotConfirmed
                    }, cancellationToken);
                    throw new ExternalServiceException("خطا در شارژ کیف پول ، در صورت کسر وجه مبلغ طی 24 ساعت به حساب شما برگشت داده خواهد شد");
                }

                var responsePurchasesZarinpalPPGCancelLink = await _httpRequestZarinLink.cancelZarinLink(new RequestCancelZarinLink
                {
                    id = command.PurchaseId
                }
, webServiceInfo.BaseUrlService, paymentLink.AccessToken, cancellationToken);

                dbResultUpdateTransaction = await _ITransActionContext.Update_Transaction(new Update_Transaction.Inputs
                {
                    TransactionCode = responsePurchasesZarinLinkTransactionById.data.Session.FirstOrDefault().reference_id,
                    PaidAmount = price,
                    PaidCode = command.PurchaseId,
                    Status = command.Status,
                    TrackingCode = trackingCode,
                    MaximummPrice = maximumAmountOnlinePort
                }, cancellationToken);

                if (dbResultUpdateTransaction != null && dbResultUpdateTransaction.Data.IsDuplicate == true)
                {
                    return _IAutomaticOrderOperationService.getPPGResult(new GetDbResultPPGCommand
                    {
                        Status = command.Status,
                        Setting = setting,
                        FinalAmount = price,
                        OrderId = dbResultUpdateTransaction.Data.OrderId,
                        TrackingCode = trackingCode,
                        CompanyId = dbResultUpdateTransaction.Data.CompanyOwnerId,
                        TransactionCode = dbResultUpdateTransaction.Data.TransactionCode,
                        PaidCode = dbResultUpdateTransaction.Data.PaidCode,
                        PaidDate = dbResultUpdateTransaction.Data.PaidDate
                    });
                }

                if (dbResultUpdateTransaction.Data.StatusOrder != null && dbResultUpdateTransaction.Data.StatusOrder == 192)
                {
                    return await _IAutomaticOrderOperationService.DoCompeleteScenario(new DoCompeleteScenarioCommand
                    {
                        OrderNumber = dbResultUpdateTransaction.Data.OrderNumber,
                        Email = dbResultUpdateTransaction.Data.Email,
                        Mobile = dbResultUpdateTransaction.Data.Mobile,
                        OrderId = dbResultUpdateTransaction.Data.OrderId,
                        OrderType = dbResultUpdateTransaction.Data.OrderType,
                        RemainingAmount = dbResultUpdateTransaction.Data.RemainingAmount,
                        StatusOrder = dbResultUpdateTransaction.Data.StatusOrder,
                        FinalAmount = dbResultUpdateTransaction.Data.FinalAmount,
                        Status = "OK",
                        CompanyId = dbResultUpdateTransaction.Data.CompanyOwnerId,
                        TrackingCode = trackingCode,
                        Setting = setting,
                        PaidCode = dbResultUpdateTransaction.Data.PaidCode,
                        TransactionCode = dbResultUpdateTransaction.Data.TransactionCode,
                        PaidDate = dbResultUpdateTransaction.Data.PaidDate,
                        ModuleType = dbResultUpdateTransaction.Data.SaveLocation
                    }, cancellationToken);
                }

                return await _IAutomaticOrderOperationService.doCompelete(new DoCompeleteCommand
                {
                    Email = dbResultUpdateTransaction.Data.Email,
                    Mobile = dbResultUpdateTransaction.Data.Mobile,
                    OrderId = dbResultUpdateTransaction.Data.OrderId,
                    OrderNumber = dbResultUpdateTransaction.Data.OrderNumber,
                    OrderType = dbResultUpdateTransaction.Data.OrderType
,
                    PurchaseId = command.PurchaseId,
                    ValueType = dbResultUpdateTransaction.Data.ValueType,
                    Status = "OK",
                    StatusOrder = dbResultUpdateTransaction.Data.StatusOrder,
                    Commission = dbResultUpdateTransaction.Data.Commission,
                    EmailCompanyUser = dbResultUpdateTransaction.Data.EmailCompanyUser,
                    MobileCompanyUser = dbResultUpdateTransaction.Data.MobileCompanyUser,
                    TransactionType = dbResultUpdateTransaction.Data.TransactionType,
                    CooperRewardToman = dbResultUpdateTransaction.Data.CooperRewardToman,
                    NumberOfSuccessfulOrders = dbResultUpdateTransaction.Data.NumberOfSuccessfulOrders,
                    FinalAmount = price.ToString(),
                    TrackingCode = trackingCode,
                    CompanyId = dbResultUpdateTransaction.Data.CompanyOwnerId,
                    Setting = setting,
                    PaidCode = dbResultUpdateTransaction.Data.PaidCode,
                    TransactionCode = dbResultUpdateTransaction.Data.TransactionCode,
                    PaidDate = dbResultUpdateTransaction.Data.PaidDate,
                }, cancellationToken);


            }

            return _IAutomaticOrderOperationService.getPPGResult(new GetDbResultPPGCommand
            {
                Status = command.Status,
                Setting = setting,
                OrderId = dbResultGetTransactionOrder.Data.OrderId,
                CompanyId = dbResultCheckTransaction.Data.CompanyId,
                TrackingCode = dbResultCheckTransaction.Data.TrackingCode,
                TransactionCode = dbResultCheckTransaction.Data.TransactionCode,
                PaidCode = dbResultCheckTransaction.Data.PaidCode,
                PaidDate = dbResultCheckTransaction.Data.PaidDate
            });

        }

        public async Task<DBResult> verifyChargeWallet(VerifyPayChargeCommand command, CancellationToken cancellationToken)
        {
            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGZarinlink, cancellationToken);

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var dbResultGetTransactionOrder = await _ITransActionContext.Get_Order_Transaction_WithouToken(new Get_Order_Transaction_WithouToken.Inputs
            {
                PaidCode = command.PurchaseId
            }, cancellationToken);
            decimal paidAmountOrder = dbResultGetTransactionOrder.Data.PaidAmount.Value;
            command.PaidAmount = Math.Round(dbResultGetTransactionOrder.Data.PaidAmount.Value).ToString();
            decimal price = command.PaidAmount.cleanPrice();

            var dbResultCheckTransaction = await _ITransActionContext.Check_Duplicate_Transaction_Wallet(new Check_Duplicate_Transaction_Wallet.Inputs
            {
                PaidCode = command.PurchaseId,
                Status = command.Status
            }, cancellationToken);

            if (dbResultCheckTransaction != null && dbResultCheckTransaction.Data != null && dbResultCheckTransaction.Data.IsDuplicate == true)
            {
                return _IAutomaticOrderOperationService.getPPGResult(new GetDbResultPPGCommand
                {
                    Status = "OK",
                    Setting = setting,
                    CompanyId = dbResultCheckTransaction.Data.CompanyId,
                    TrackingCode = dbResultCheckTransaction.Data.TrackingCode,
                    TransactionCode = dbResultCheckTransaction.Data.TransactionCode,
                    PaidCode = dbResultCheckTransaction.Data.PaidCode,
                    PaidDate = dbResultCheckTransaction.Data.PaidDate,
                    FinalAmount = dbResultCheckTransaction.Data.PaidAmount,
                    WalletId = dbResultCheckTransaction.Data.WalletId
                });
            }

            if (command.Status == "OK")
            {
                var responsePaymentLink = await _paymentLinkContext.GetPaymentLink_WithoutToken(
                     new GetPaymentLink_WithoutToken.Inputs { ExternalId = command.PurchaseId }, cancellationToken);
                var paymentLink = responsePaymentLink.Data.First();
                var responsePurchasesZarinpalPPG = await _httpRequestZarinLink.getTransactionListByZarinLinkId(new RequestGetTransactionZarinLink
                {
                    terminal_id = paymentLink.Terminal
                            ,
                    relation_id = command.PurchaseId
                }
                            , webServiceInfo.BaseUrlService, paymentLink.AccessToken, cancellationToken);

                if (responsePurchasesZarinpalPPG == null || responsePurchasesZarinpalPPG.data == null
    || responsePurchasesZarinpalPPG.data.Session == null
    || !responsePurchasesZarinpalPPG.data.Session.Any(q => q.status.Equals("PAID")))
                {

                    throw new ExternalServiceException("خطا در شارژ کیف پول ، در صورت کسر وجه مبلغ طی 24 ساعت به حساب شما برگشت داده خواهد شد");
                }

                var responsePurchasesZarinLinkById = await _httpRequestZarinLink.getZarinLinkById(new RequestGetZarinLink
                {
                    terminal_id = paymentLink.Terminal,
                    id = command.PurchaseId
                }

, webServiceInfo.BaseUrlService, paymentLink.AccessToken, cancellationToken);
                if (responsePurchasesZarinLinkById == null || responsePurchasesZarinLinkById.data == null ||
                  responsePurchasesZarinLinkById.data.NewZarinLinks == null || responsePurchasesZarinLinkById.data.NewZarinLinks.Count == 0 || responsePurchasesZarinLinkById.data.NewZarinLinks.FirstOrDefault().deleted_at != null)
                {

                    throw new ExternalServiceException("خطا در شارژ کیف پول ، در صورت کسر وجه مبلغ طی 24 ساعت به حساب شما برگشت داده خواهد شد");
                }
                var trackingCode = responsePurchasesZarinpalPPG?.data?.Session?.FirstOrDefault(q => q.status.Equals("PAID"))?.id;
                var responsePurchasesZarinLinkTransactionById = await _httpRequestZarinLink.getTransactionByIdByZarinLinkId(new RequestGetTransactionZarinLink
                {
                    terminal_id = paymentLink.Terminal,
                    relation_id = trackingCode
                }

                , webServiceInfo.BaseUrlService, paymentLink.AccessToken, cancellationToken);

                if (responsePurchasesZarinLinkTransactionById == null || responsePurchasesZarinLinkTransactionById.data == null ||
                responsePurchasesZarinLinkTransactionById.data.Session == null || responsePurchasesZarinLinkTransactionById.data.Session.Count == 0)
                {

                    throw new ExternalServiceException("خطا در شارژ کیف پول ، در صورت کسر وجه مبلغ طی 24 ساعت به حساب شما برگشت داده خواهد شد");
                }


                var responsePurchasesZarinpalPPGCancelLink = await _httpRequestZarinLink.cancelZarinLink(new RequestCancelZarinLink
                {
                    id = command.PurchaseId
                }
, webServiceInfo.BaseUrlService, paymentLink.AccessToken, cancellationToken);

                var dbResultUpdateTransaction = await _ITransActionContext.Update_Transaction_Wallet(new Update_Transaction_Wallet.Inputs
                {
                    TransactionCode = responsePurchasesZarinLinkTransactionById.data.Session.FirstOrDefault().reference_id,
                    PaidAmount = price,
                    PaidCode = command.PurchaseId,
                    Status = command.Status,
                    TrackingCode = trackingCode
                }, cancellationToken);

                if (dbResultUpdateTransaction != null && dbResultUpdateTransaction.Data != null && dbResultUpdateTransaction.Data.IsDuplicate == true)
                {
                    return _IAutomaticOrderOperationService.getPPGResult(new GetDbResultPPGCommand
                    {
                        Status = "OK",
                        FinalAmount = price,
                        TrackingCode = trackingCode,
                        Setting = setting,
                        CompanyId = dbResultUpdateTransaction.Data.CompanyId,
                        TransactionCode = dbResultUpdateTransaction.Data.TransactionCode,
                        PaidCode = dbResultUpdateTransaction.Data.PaidCode,
                        PaidDate = dbResultUpdateTransaction.Data.PaidDate,
                        WalletId = dbResultUpdateTransaction.Data.WalletId
                    });
                }

                return await _IAutomaticOrderOperationService.doCompeleteWalletCharge(
                new DoCompeleteWalletCharge
                {
                    PaidAmount = dbResultUpdateTransaction.Data.PaidAmount,
                    Email = dbResultUpdateTransaction.Data.Email,
                    Mobile = dbResultUpdateTransaction.Data.Mobile,
                    Status = "OK",
                    TrackingCode = trackingCode,
                    CompanyId = dbResultUpdateTransaction.Data.CompanyId,
                    Setting = setting,
                    PaidCode = dbResultUpdateTransaction.Data.PaidCode,
                    TransactionCode = dbResultUpdateTransaction.Data.TransactionCode,
                    PaidDate = dbResultUpdateTransaction.Data.PaidDate,
                    WalletId = dbResultUpdateTransaction.Data.WalletId
                }, cancellationToken);



            }

            return _IAutomaticOrderOperationService.getPPGResult(new GetDbResultPPGCommand
            {
                Status = command.Status,
                Setting = setting,
                CompanyId = dbResultCheckTransaction.Data.CompanyId,
                TrackingCode = dbResultCheckTransaction.Data.TrackingCode,
                TransactionCode = dbResultCheckTransaction.Data.TransactionCode,
                PaidCode = dbResultCheckTransaction.Data.PaidCode,
                PaidDate = dbResultCheckTransaction.Data.PaidDate,
                WalletId = dbResultCheckTransaction.Data.WalletId
            });
        }

        public async Task<DBResult> getZarinLinkHistory(InquiryZarinLinkCommand command, CancellationToken cancellationToken)
        {

            string token = _tokenUserService.getUserInfo().Token;

            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGZarinlink, cancellationToken);

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var response = await _paymentLinkContext.GetPaymentLink(new GetPaymentLink.Inputs
            {
                Token = token,
                PaymentLinkId = command.PaymentLinkId,
            }, cancellationToken);

            var responseGetZarinLink = new ResponseGetZarinLink();
            responseGetZarinLink = await _httpRequestZarinLink.getZarinLink(new RequestGetZarinLink
            {
                terminal_id = response.Data.FirstOrDefault().Terminal,
                limit = command.PageSize,
                offset = (command.PageSize * (command.StartIndex - 1))
            }
            , webServiceInfo.BaseUrlService, response.Data.FirstOrDefault().AccessToken, cancellationToken);

            return new DBResult(responseGetZarinLink, 1);

        }
        public async Task<DBResult> getZarinLinkTransaction(InquiryZarinLinkTransaction command, CancellationToken cancellationToken)
        {
            string token = _tokenUserService.getUserInfo().Token;
            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGZarinlink, cancellationToken);


            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var response = await _paymentLinkContext.GetPaymentLinkHistory(new GetPaymentLinkHistory.Inputs
            {
                ExternalId = command.ExternalId,
                Token = token,
            }, cancellationToken);
            if (response == null || response.Data == null || !response.Data.Any())
            {
                throw new LogicalException("شناسه پرداخت در سیستم یافت نشد");
            }
            var responseGetZarinLink = await _httpRequestZarinLink.getTransactionListByZarinLinkId(new RequestGetTransactionZarinLink
            {
                terminal_id = response.Data.FirstOrDefault().Terminal,
                limit = command.PageSize,
                offset = (command.PageSize * (command.StartIndex - 1)),
                filter = command.Filter,
                relation_id = command.ExternalId
            }
            , webServiceInfo.BaseUrlService, response.Data.FirstOrDefault().AccessToken, cancellationToken);

            return new DBResult(responseGetZarinLink, 1);

        }

        public async Task<DBResult> getBankAccount(InquiryZarinLinkByIdCommand command, CancellationToken cancellationToken)
        {
            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGZarinlink, cancellationToken);

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var response = await _paymentLinkContext.GetPaymentLinkHistory(new GetPaymentLinkHistory.Inputs
            {
                ExternalId = command.Id
            }, cancellationToken);

            var responseGetZarinLink = await _httpRequestZarinLink.getBankAccount(webServiceInfo.BaseUrlService, response.Data.FirstOrDefault().AccessToken, cancellationToken);

            return new DBResult(responseGetZarinLink, 1);

        }

        public async Task<DBResult> settlement(SettlementZarinLinkSettingCommand command, CancellationToken cancellationToken)
        {
            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGZarinlink, cancellationToken);

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var response = await _paymentLinkContext.GetPaymentLinkHistory(new GetPaymentLinkHistory.Inputs
            {
                ExternalId = command.ExternalId
            }, cancellationToken);

            var responseGetZarinLink = await _httpRequestZarinLink.settlementZarinLinkSetting(new RequestSettlementZarinLink
            {
                bank_account_id = command.BankAccountId,
                id = response.Data.FirstOrDefault().Terminal,
                reconcile_priority = command.ReconcilePriority,
            }, webServiceInfo.BaseUrlService, response.Data.FirstOrDefault().AccessToken, cancellationToken);

            return new DBResult(responseGetZarinLink, 1);

        }

        public async Task<DBResult<ResponseGetPayZarinLink>> getPay(GetPayShepaCommand command, CancellationToken cancellationToken, bool checkJobToken = true)
        {
            if (checkJobToken)
            {
                var dbSetting = await _ISettingContext.Get_Setting_Job(new Get_Setting_Job.Inputs { }, cancellationToken);
                if (dbSetting.Data.JobToken != command.JobToken)
                {
                    throw new LogicalException("کلید اتصال نامعتبر می باشد");
                }
            }

            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGZarinlink, cancellationToken);


            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var dbResultGetTransactionOrder = await _ITransActionContext.Get_Order_Transaction_WithouToken(new Get_Order_Transaction_WithouToken.Inputs
            {
                PaidCode = command.PurchaseId
            }, cancellationToken);

            command.PaidAmount = Math.Round(dbResultGetTransactionOrder.Data.PaidAmount.Value).ToString();
            decimal price = 0;
            if (command.PaidAmount.ToString().Contains("."))
                price = Convert.ToDecimal(command.PaidAmount.ToString().Split(".")[0]);
            else
                price = Convert.ToDecimal(command.PaidAmount);

            var responsePaymentLink = await _paymentLinkContext.GetPaymentLink_WithoutToken(
                   new GetPaymentLink_WithoutToken.Inputs { ExternalId = command.PurchaseId }, cancellationToken);
            var paymentLink = responsePaymentLink.Data.First();
            var responsePurchasesZarinpalPPG = await _httpRequestZarinLink.getTransactionListByZarinLinkId(new RequestGetTransactionZarinLink
            {
                terminal_id = paymentLink.Terminal
                         ,
                relation_id = command.PurchaseId
            }
                         , webServiceInfo.BaseUrlService, paymentLink.AccessToken, cancellationToken);

            if (responsePurchasesZarinpalPPG == null || responsePurchasesZarinpalPPG.data == null
|| responsePurchasesZarinpalPPG.data.Session == null
|| !responsePurchasesZarinpalPPG.data.Session.Any(q => q.status.Equals("PAID")))
            {
                return new DBResult<ResponseGetPayZarinLink>(new ResponseGetPayZarinLink { }, 1);
            }
            var responsePurchasesZarinLinkById = await _httpRequestZarinLink.getZarinLinkById(new RequestGetZarinLink
            {
                terminal_id = paymentLink.Terminal,
                id = command.PurchaseId
            }

, webServiceInfo.BaseUrlService, paymentLink.AccessToken, cancellationToken);
            if (responsePurchasesZarinLinkById == null || responsePurchasesZarinLinkById.data == null ||
              responsePurchasesZarinLinkById.data.NewZarinLinks == null || responsePurchasesZarinLinkById.data.NewZarinLinks.Count == 0 || responsePurchasesZarinLinkById.data.NewZarinLinks.FirstOrDefault().deleted_at != null)
            {
                return new DBResult<ResponseGetPayZarinLink>(new ResponseGetPayZarinLink { }, 1);
            }
            var trackingCode = responsePurchasesZarinpalPPG?.data?.Session?.FirstOrDefault(q => q.status.Equals("PAID"))?.id;
            var responsePurchasesZarinLinkTransactionById = await _httpRequestZarinLink.getTransactionByIdByZarinLinkId(new RequestGetTransactionZarinLink
            {
                terminal_id = paymentLink.Terminal,
                relation_id = trackingCode
            }
            , webServiceInfo.BaseUrlService, paymentLink.AccessToken, cancellationToken);

            if (responsePurchasesZarinLinkTransactionById == null || responsePurchasesZarinLinkTransactionById.data == null ||
            responsePurchasesZarinLinkTransactionById.data.Session == null || responsePurchasesZarinLinkTransactionById.data.Session.Count == 0)
            {
                return new DBResult<ResponseGetPayZarinLink>(new ResponseGetPayZarinLink { }, 1);
            }


            var responsePurchasesZarinpalPPGCancelLink = await _httpRequestZarinLink.cancelZarinLink(new RequestCancelZarinLink
            {
                id = command.PurchaseId
            }
, webServiceInfo.BaseUrlService, paymentLink.AccessToken, cancellationToken);


            return new DBResult<ResponseGetPayZarinLink>(new
            ResponseGetPayZarinLink
            {
                PaidAmount = dbResultGetTransactionOrder.Data.PaidAmount.Value
                ,
                TrackingCode = trackingCode
                ,
                TransactionCode = responsePurchasesZarinLinkTransactionById.data?.Session?.FirstOrDefault()?.reference_id
            }, 1);

        }

        public async Task<DBResult> GetTransactionDetails(GetTransactionDetailsCommand command, CancellationToken cancellationToken)
        {
            string token = _tokenUserService.getUserInfo().Token;
            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGZarinlink, cancellationToken);

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var transaction = await _ITransActionContext.GetTransActionsByAdmin(new GetTransActionsByAdmin.Inputs
            {
                Token = token,
                PaidCode = command.ExternalId
            }, cancellationToken);
            if (transaction == null || transaction.Data == null || !transaction.Data.Any(q => !string.IsNullOrWhiteSpace(q.TrackingCode)))
            {
                throw new LogicalException("تراکنش مورد نظر کد پیگیری ندارد");
            }
            var paymentLink = await _paymentLinkContext.GetPaymentLink_WithoutToken(new GetPaymentLink_WithoutToken.Inputs
            {
                ExternalId = command.ExternalId,
            }, cancellationToken);
            if (paymentLink == null || paymentLink.Data == null || !paymentLink.Data.Any())
            {
                throw new LogicalException("کد پرداخت ارسال شده متعلق به حساب زرین لینک نمی باشد ، ممکن است حساب زرین لینک مورد نظر حذف شده باشد");
            }

            var response = await _httpRequestZarinLink.getTransactionByIdByZarinLinkId(new RequestGetTransactionZarinLink
            {
                terminal_id = paymentLink.Data.FirstOrDefault().Terminal,
                relation_id = transaction.Data.FirstOrDefault().TrackingCode
            }, webServiceInfo.BaseUrlService, paymentLink.Data.FirstOrDefault().AccessToken, cancellationToken);

            return new DBResult(response, 1);
        }

        public async Task<DBResult> InquiryCardHolder(InquiryCardHolderZarinLinkCommand command, CancellationToken cancellationToken)
        {
            string token = _tokenUserService.getUserInfo().Token;
            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGZarinlink, cancellationToken);


            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var transaction = await _ITransActionContext.GetTransActionsByAdmin(new GetTransActionsByAdmin.Inputs
            {
                Token = token,
                PaidCode = command.ExternalId
            }, cancellationToken);
            if (transaction == null || transaction.Data == null || !transaction.Data.Any(q => !string.IsNullOrWhiteSpace(q.TrackingCode)))
            {
                throw new LogicalException("تراکنش مورد نظر کد پیگیری ندارد");
            }

            var paymentLink = await _paymentLinkContext.GetPaymentLink_WithoutToken(new GetPaymentLink_WithoutToken.Inputs
            {
                ExternalId = command.ExternalId,
            }, cancellationToken);
            if (paymentLink == null || paymentLink.Data == null || !paymentLink.Data.Any())
            {
                throw new LogicalException("کد پرداخت ارسال شده متعلق به حساب زرین لینک نمی باشد ، ممکن است حساب زرین لینک مورد نظر حذف شده باشد");
            }


            var response = await _httpRequestZarinLink.SetCardHolder(transaction.Data.FirstOrDefault().TrackingCode, webServiceInfo.BaseUrlService, paymentLink.Data.FirstOrDefault().AccessToken, cancellationToken);
            if (response != null && response.data != null && response.data.SessionCardHolder == true)
            {
                return new DBResult(new { });
            }
            throw new ExternalServiceException("عملیات استعلام صاحب حساب باخطا مواجه شد");

        }

        public async Task<DBResult> AddRefund(AddRefundZarinLinkCommand command, CancellationToken cancellationToken)
        {
            string token = _tokenUserService.getUserInfo().Token;
            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGZarinlink, cancellationToken);

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var transaction = await _ITransActionContext.GetTransActionsByAdmin(new GetTransActionsByAdmin.Inputs
            {
                Token = token,
                PaidCode = command.ExternalId
            }, cancellationToken);
            if (transaction == null || transaction.Data == null || !transaction.Data.Any(q => !string.IsNullOrWhiteSpace(q.TrackingCode)))
            {
                throw new LogicalException("تراکنش مورد نظر کد پیگیری ندارد");
            }

            var paymentLink = await _paymentLinkContext.GetPaymentLink_WithoutToken(new GetPaymentLink_WithoutToken.Inputs
            {
                ExternalId = command.ExternalId,
            }, cancellationToken);
            if (paymentLink == null || paymentLink.Data == null || !paymentLink.Data.Any())
            {
                throw new LogicalException("کد پرداخت ارسال شده متعلق به حساب زرین لینک نمی باشد ، ممکن است حساب زرین لینک مورد نظر حذف شده باشد");
            }


            var response = await _httpRequestZarinLink.AddRefund(new RequestRefundZarinLink
            {
                session_id = transaction.Data.FirstOrDefault().TrackingCode,
                description = command.Description,
                amount = Convert.ToInt64(transaction.Data.FirstOrDefault().PaidAmount)

            }, webServiceInfo.BaseUrlService, paymentLink.Data.FirstOrDefault().AccessToken, cancellationToken);
            if (response != null && response.data != null && response.data.AddRefund != null
                && !string.IsNullOrWhiteSpace(response.data.AddRefund.id))
            {
                return new DBResult(new { });
            }
            throw new ExternalServiceException("عملیات درخواست استرداد وجه باخطا مواجه شد");

        }

        public async Task<DBResult> GetCurrentBalance(GetCurrentBalanceCommand command, CancellationToken cancellationToken)
        {
            var setting = await _redisContextService.GetSettingPurchaseAsync(cancellationToken);
            var webServiceInfo = await _redisContextService.GetWebserviceAsync(WebServiceType.PPGZarinlink, cancellationToken);

            var config = webServiceInfo.AdditionalConfig.JsonParse();
            var paymentLink = await _paymentLinkContext.GetPaymentLink_WithoutToken(new GetPaymentLink_WithoutToken.Inputs
            {
                Terminal = command.TerminalId,
            }, cancellationToken);
            if (paymentLink == null || paymentLink.Data == null || !paymentLink.Data.Any())
            {
                throw new LogicalException("ترمینال ارسال شده متعلق به حساب زرین لینک نمی باشد ، ممکن است حساب زرین لینک مورد نظر حذف شده باشد");
            }
            var response = await _httpRequestZarinLink.GetCurrentBalance(command.TerminalId, webServiceInfo.BaseUrlService, paymentLink.Data.FirstOrDefault().AccessToken, cancellationToken);
            if (response != null && response.data != null && response.data.PayoutBalance != null)
            {
                return new DBResult(response, 1);
            }
            throw new ExternalServiceException("عملیات استعلام موجودی حساب باخطا مواجه شد");

        }


        private async Task<GetPaymentLink_WithoutToken.Outputs> getActiveZarinLink(decimal? finalAmountToman, CancellationToken cancellationToken)
        {
            var zarinLinkAccount = await _paymentLinkContext.GetPaymentLink_WithoutToken(new GetPaymentLink_WithoutToken.Inputs
            {

            }, cancellationToken);
            if (zarinLinkAccount == null || zarinLinkAccount.Data == null || zarinLinkAccount.Data.Count() == 0)
            {
                throw new LogicalException("ترمینال فعالی جهت پرداخت آنلاین یافت نشد ، در صورت تکرار خطا مراتب را به کارشناسان ما اطلاع بدهید");
            }
            var zarinLink = zarinLinkAccount.Data.FirstOrDefault(q => (q.MaximumNumberOfTransactionsPerMonth == null || q.MaximumNumberOfTransactionsPerMonth <= 0 || q.MaximumNumberOfTransactionsPerMonth > q.history_count)
            && (q.MaximumTransactionAmountPerMonth == null || q.MaximumTransactionAmountPerMonth <= 0 || q.MaximumTransactionAmountPerMonth > (q.history_amount + finalAmountToman)));
            if (zarinLink == null)
            {

                throw new AccountLimitationException("تمام حساب های بانکی به سقف مجاز رسیده اند ، لطفااز درگاه پرداخت دیگری استفاده نمایید ، درصورت تکرار خطا می توانید با کارشناسان ما ارتباط برقرار کنید");
            }
            return zarinLink;
        }
    }
}
