﻿using Farakonesh.Logic.CommonOperations;
using Farakonesh.Logic.IDatabase.idbo;
using Farakonesh.Logic.IDatabase.IOrder;
using Farakonesh.Logic.IDatabase.IUser;
using Farakonesh.Logic.IServices.ICache;
using Farakonesh.Models.Database.StoredProcedures.App.dbo.keys;
using Farakonesh.Models.Database.StoredProcedures.App.dbo.setting;
using Farakonesh.Models.Database.StoredProcedures.App.Order;
using Farakonesh.Models.Database.StoredProcedures.App.User.User;
using Farakonesh.Models.Database.Systems;
using Farakonesh.Shared.Enums;
using Farakonesh.Shared.Exceptions;
using Microsoft.Extensions.Caching.Memory;
using Microsoft.Extensions.Configuration;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using static System.Runtime.InteropServices.JavaScript.JSType;

namespace Farakonesh.Logic.Services.Cache
{
    public class RedisContextService : IRedisContextService
    {
        private readonly RedisService _redis;
        private readonly IMemoryCache _cache;
        private readonly ISettingContext _settingContext;
        private readonly IServiceKeysContext _serviceKeysContext;
        private readonly IPublicContext _publicContext;
        private readonly IUserContext _userContext;
        private readonly ICurrencyContext _currencyContext;
        private readonly ICountryContext _countryContext;
        private CryptoHelper _crypto;
        public RedisContextService(RedisService redis, ISettingContext settingContext
            , IPublicContext publicContext, IUserContext userContext, ICurrencyContext currencyContext
            , ICountryContext countryContext, IMemoryCache cache, IServiceKeysContext serviceKeysContext, IConfiguration configuration)
        {
            _crypto = new CryptoHelper(configuration["CryptoSettings:AesKey"]);
            _redis = redis;
            _settingContext = settingContext;
            _serviceKeysContext = serviceKeysContext;
            _publicContext = publicContext;
            _userContext = userContext;
            _currencyContext = currencyContext;
            _countryContext = countryContext;
            _cache = cache;
        }


        public async Task<Get_Setting_Purchase_Verify.Outputs> GetSettingPurchaseAsync(CancellationToken cancellationToken)
        {
            var setting = await _redis.Get<Get_Setting_Purchase_Verify.Outputs>("settingPurchase");
            if (setting == null)
            {
                var dataSetting = await _settingContext.Get_Setting_Purchase_Verify(new Get_Setting_Purchase_Verify.Inputs(), cancellationToken);
                setting = await _redis.Set<Get_Setting_Purchase_Verify.Outputs>("settingPurchase", dataSetting.Data);
            }
            return setting;
        }

        public async Task<Get_Order_Setting_WithoutToken.Outputs> getSettingOrderWithoutTokenAsync(int? orderType, Guid? orderId, CancellationToken cancellationToken)
        {
            var setting = await _redis.Get<Get_Order_Setting_WithoutToken.Outputs>("settingOrderWithoutToken");
            if (setting == null)
            {
                var dataOrderSetting = await _publicContext.Get_Order_Setting_WithoutToken(new Get_Order_Setting_WithoutToken.Inputs
                {
                    IsArchive = false,
                    OrderType = orderType,
                    OrderId = orderId
                }, cancellationToken);
                setting = await _redis.Set<Get_Order_Setting_WithoutToken.Outputs>("settingOrderWithoutToken", dataOrderSetting.Data);
            }
            return setting;
        }

        public async Task<Get_Order_Setting.Outputs> getSettingOrderAsync(int? orderType, string token, CancellationToken cancellationToken)
        {
            string key = "settingOrder";
            if (orderType != null && orderType > 0)
            {
                key = "settingOrder_" + orderType;
            }
            var setting = await _redis.Get<Get_Order_Setting.Outputs>(key);
            if (setting == null)
            {
                var dataOrderSetting = await _publicContext.Get_Order_Setting(new Get_Order_Setting.Inputs
                {
                    IsArchive = false,
                    OrderType = orderType,
                    Token = token
                }, cancellationToken);

                setting = await _redis.Set<Get_Order_Setting.Outputs>(key, dataOrderSetting.Data);
            }
            return setting;
        }


        public async Task<Get_Setting_Basic_System.Outputs> GetSettingBasicSystemAsync(CancellationToken cancellationToken)
        {
            var setting = await _redis.Get<Get_Setting_Basic_System.Outputs>("settingBasic");
            if (setting == null)
            {
                var rst = await _settingContext.Get_Setting_Basic_System(new Get_Setting_Basic_System.Inputs { }, cancellationToken);
                setting = await _redis.Set<Get_Setting_Basic_System.Outputs>("settingBasic", rst.Data);

            }
            return setting;
        }

        public async Task<Get_Setting_SMS.Outputs> getSettingSmsAsync(CancellationToken cancellationToken)
        {
            var setting = await _redis.Get<Get_Setting_SMS.Outputs>("settingSms");
            if (setting == null)
            {
                var dataSetting = await _settingContext.Get_Setting_SMS(new Get_Setting_SMS.Inputs { }, cancellationToken);
                setting = await _redis.Set<Get_Setting_SMS.Outputs>("settingSms", dataSetting.Data);
            }
            return setting;
        }

        public async Task<ResponseCache> GetWebserviceAsync(
      WebServiceType webServiceType,
      CancellationToken cancellationToken)
        {
            string key = "webService_" + (int)webServiceType;

            _cache.TryGetValue<SensitiveKeyData>(key, out var sensitive);
            var meta = await _redis.Get<WebserviceMetaData>(key);
            var realPassword = "";
            var realApiKey = "";
            if (sensitive != null && meta != null)
            {
                realPassword = _crypto.DecryptApiKey(sensitive.EncryptedPassword, sensitive.EncryptionPasswordIV);
                realApiKey = _crypto.DecryptApiKey(sensitive.EncryptedApiKey, sensitive.EncryptionKeyIV);
                return ResponseCacheExtensions.Merge(meta, realApiKey, realPassword);
            }


            var db = await _serviceKeysContext.Get_ServiceKeys(new Get_ServiceKeys.Inputs
            {
                KeysId = null,
                KeyType = (int)webServiceType
            }, cancellationToken);
            if (db == null || db.Data == null || db.Data.Count() <= 0)
            {
                throw new LogicalException("تنظیمات وب سرویس یافت نشد، در صورت تکرار خطا لطفا به پشتیبانی سیستم اطلاع دهید");
            }
            var d = db.Data.FirstOrDefault();
            if (d.IsVisible == null || d.IsVisible == false)
            {
                return null;
            }
            meta = new WebserviceMetaData
            {
                ServiceKeysId = d.ServiceKeysId,
                Title = d.Title,
                Username = d.Username,
                BaseUrlService = d.BaseUrlService,
                AdditionalConfig = d.AdditionalConfig,
                NumberRequestsPerHour = d.NumberRequestsPerHour,
                NumberRequestsPerDay = d.NumberRequestsPerDay,
                NumberRequestsPerMonth = d.NumberRequestsPerMonth,
                NumberRequests = d.NumberRequests,
                NumberRequestsMade = d.NumberRequestsMade,
                SaveDate = d.SaveDate,
                ExpireDate = d.ExpireDate,
                LastRequestDate = d.LastRequestDate,
                ServiceKeyType = d.ServiceKeyType
            };

            sensitive = new SensitiveKeyData
            {
                EncryptedApiKey = d.EncryptedApiKey,
                EncryptedPassword = d.EncryptedPassword,
                EncryptionKeyIV = d.EncryptionKeyIV,
                EncryptionPasswordIV = d.EncryptionPasswordIV,
                ApiKeyHash = d.ApiKeyHash,
                PasswordHash = d.PasswordHash
            };

            await _redis.Set<object>(key, meta);

            _cache.Set(key, sensitive, new MemoryCacheEntryOptions
            {
                SlidingExpiration = TimeSpan.FromHours(6),
                AbsoluteExpirationRelativeToNow = TimeSpan.FromHours(12)
            });
            realPassword = _crypto.DecryptApiKey(sensitive.EncryptedPassword, sensitive.EncryptionPasswordIV);
            realApiKey = _crypto.DecryptApiKey(sensitive.EncryptedApiKey, sensitive.EncryptionKeyIV);
            return ResponseCacheExtensions.Merge(meta, realApiKey, realPassword);
        }

        public async Task<Get_Setting_Telegram.Outputs> GetSettingTelegramAsync(CancellationToken cancellationToken)
        {
            var setting = await _redis.Get<Get_Setting_Telegram.Outputs>("settingTelegram");
            if (setting == null)
            {
                var dataSetting = await _settingContext.Get_Setting_Telegram(new Get_Setting_Telegram.Inputs { }, cancellationToken);
                setting = await _redis.Set<Get_Setting_Telegram.Outputs>("settingTelegram", dataSetting.Data);
            }
            return setting;
        }


        public async Task<Get_Setting_Email.Outputs> GetSettingEmailAsync(CancellationToken cancellationToken)
        {
            var setting = await _redis.Get<Get_Setting_Email.Outputs>("settingEmail");
            if (setting == null)
            {
                var dataSetting = await _settingContext.Get_Setting_Email(new Get_Setting_Email.Inputs { }, cancellationToken);
                setting = await _redis.Set<Get_Setting_Email.Outputs>("settingEmail", dataSetting.Data);
            }
            return setting;
        }


        public async Task<Get_Setting_Auth.Outputs> GetSettingAuthAsync(CancellationToken cancellationToken)
        {
            var setting = await _redis.Get<Get_Setting_Auth.Outputs>("settingAuth");
            if (setting == null)
            {
                var dataSetting = await _settingContext.Get_Setting_Auth(new Get_Setting_Auth.Inputs { }, cancellationToken);

                setting = await _redis.Set<Get_Setting_Auth.Outputs>("settingAuth", dataSetting.Data);
            }
            if (setting != null && setting.IsActiveGoogleRecaptcha == true)
            {
                var webServiceInfo = await GetWebserviceAsync(WebServiceType.GoogleRecaptcha, cancellationToken);
                setting.SecretKeyGoogleRecaptcha = webServiceInfo.RealApiKey;
            }
            return setting;
        }


        public async Task<Get_Setting_WindowsService.Outputs> GetSettingWindowsServiceAsync(CancellationToken cancellationToken)
        {
            var setting = await _redis.Get<Get_Setting_WindowsService.Outputs>("settingWindowsService");
            if (setting == null)
            {
                var dataSetting = await _settingContext.Get_Setting_WindowsService(new Get_Setting_WindowsService.Inputs { }, cancellationToken);
                setting = await _redis.Set<Get_Setting_WindowsService.Outputs>("settingWindowsService", dataSetting.Data);
            }
            return setting;
        }

        public async Task<Get_Config_User_Wallet_ByCustomer.Outputs> getSettingConfigWalletAsync(string token, CancellationToken cancellationToken)
        {
            var setting = await _redis.Get<Get_Config_User_Wallet_ByCustomer.Outputs>("configUser");
            if (setting == null)
            {
                var dataSettingUser = await _userContext.Get_Config_User_Wallet_ByCustomer(new Get_Config_User_Wallet_ByCustomer.Inputs { Token = token }, cancellationToken);
                setting = await _redis.Set<Get_Config_User_Wallet_ByCustomer.Outputs>("configUser", dataSettingUser.Data);
            }
            return setting;
        }


        public async Task<List<Get_Order_SettingList_WithoutToken.Outputs>> GetOrderSettingListWithoutTokenUseInTelegramAsync(int? moduleType, CancellationToken cancellationToken)
        {
            string key = "settingOrderWithoutTokenUseInTelegramCrawler";
            if (moduleType != null && moduleType >= 0)
            {
                key = "settingOrderWithoutTokenUseInTelegramCrawler_" + moduleType;
            }
            var setting = await _redis.Get<List<Get_Order_SettingList_WithoutToken.Outputs>>(key);
            if (setting == null)
            {
                var rst = await _settingContext.Get_Order_SettingList_WithoutToken(new Get_Order_SettingList_WithoutToken.Inputs { UseInTelegram = true, ModuleType = moduleType }, cancellationToken);
                setting = await _redis.Set<List<Get_Order_SettingList_WithoutToken.Outputs>>(key, rst.Data);
            }
            return setting;
        }

        public async Task<List<GetCurrencyList.Outputs>> GetCurrencyListAsync(GetCurrencyList.Inputs inputs, CancellationToken cancellationToken)
        {
            string type = "";
            if (inputs.ShowInMessagingTelegram == null)
            {
                type = "get_free";
            }
            if (inputs.ShowInMessagingTelegram == true)
            {
                type = "get_InMessagingTelegram";
            }
            if (inputs.ShowInMessagingTelegram == false)
            {
                type = "get_NotInMessagingTelegram";
            }
            var response = await _redis.Get<List<GetCurrencyList.Outputs>>(type + "currencyList" + inputs.OrderType);
            if (response == null)
            {
                var rst = await _currencyContext.GetCurrencyList(inputs, cancellationToken);

                response = await _redis.Set<List<GetCurrencyList.Outputs>>(type + "currencyList" + inputs.OrderType, rst.Data);
            }
            return response;

        }

        public async Task<List<GetCurrencyList_Servant.Outputs>> GetCurrencyListServantAsync(GetCurrencyList_Servant.Inputs inputs, CancellationToken cancellationToken)
        {
            string type = "";
            if (inputs.ShowInMessagingTelegram == null)
            {
                type = "get_free";
            }
            if (inputs.ShowInMessagingTelegram == true)
            {
                type = "get_InMessagingTelegram";
            }
            if (inputs.ShowInMessagingTelegram == false)
            {
                type = "get_NotInMessagingTelegram";
            }
            var response = await _redis.Get<List<GetCurrencyList_Servant.Outputs>>(type + "currencyServantList" + inputs.OrderType);
            if (response == null)
            {
                var rst = await _currencyContext.GetCurrencyList_Servant(inputs, cancellationToken);

                response = await _redis.Set<List<GetCurrencyList_Servant.Outputs>>(type + "currencyServantList" + inputs.OrderType, rst.Data);
            }
            return response;

        }

        public async Task<List<GetCountryList.Outputs>> GetCountryListAsync(GetCountryList.Inputs inputs, CancellationToken cancellationToken)
        {
            if (inputs.Continent == null || inputs.Continent <= 0)
            {
                inputs.Continent = -1;
            }
            string type = "";
            if (!string.IsNullOrWhiteSpace(inputs.PrefixPhone))
            {
                type = "_get_with_prefix" + inputs.PrefixPhone;
            }
            else
            {
                if (inputs.UseInOrder == true &&
           ((inputs.UserInRegister == null || inputs.UserInRegister == false)
           && (inputs.UseInTelegram == null || inputs.UseInTelegram == false)
           && (inputs.UseInMessagingTelegram == null || inputs.UseInMessagingTelegram == false)))
                {
                    type = "_get_for_order";
                }

                if (inputs.UseInMessagingTelegram == true &&
                    ((inputs.UserInRegister == null || inputs.UserInRegister == false)
                    && (inputs.UseInTelegram == null || inputs.UseInTelegram == false)
                    && (inputs.UseInOrder == null || inputs.UseInOrder == false)))
                {
                    type = "_get_for_messagingTelegram";
                }

                if (inputs.UseInTelegram == true &&
                 ((inputs.UserInRegister == null || inputs.UserInRegister == false)
                 && (inputs.UseInOrder == null || inputs.UseInOrder == false)
                 && (inputs.UseInMessagingTelegram == null || inputs.UseInMessagingTelegram == false)))
                {
                    type = "_get_for_telegram";
                }
                if ((inputs.UseInOrder == null && inputs.UserInRegister == null && inputs.UseInTelegram == null && inputs.UseInMessagingTelegram == null)
                    ||
                    (inputs.UseInOrder == true && inputs.UserInRegister == true && inputs.UseInTelegram == true && inputs.UseInMessagingTelegram == true))
                {
                    type = "_get_for_all";
                }
                if (inputs.UserInRegister == true && ((inputs.UseInOrder == null || inputs.UseInOrder == false)
                    && (inputs.UseInTelegram == null || inputs.UseInTelegram == false)
                    && (inputs.UseInMessagingTelegram == null || inputs.UseInMessagingTelegram == false)))
                {
                    type = "_get_for_register";
                }
                if (inputs.UseInOrder == false && inputs.UserInRegister == false
                    && inputs.UseInTelegram == false && inputs.UseInMessagingTelegram == false)
                {
                    type = "_get_free";
                }
            }

            var responseCache = await _redis.Get<List<GetCountryList.Outputs>>(inputs.Continent + "_countryList" + type);
            if (responseCache == null)
            {
                var response = await _countryContext.GetCountryList(inputs, cancellationToken);
                await _redis.Set<List<GetCountryList.Outputs>>(inputs.Continent + "_countryList" + type, response.Data, 600);
                responseCache = response.Data.ToList();
            }
            return responseCache;

        }



    }
}
