﻿using Microsoft.Extensions.Configuration;
using Farakonesh.Logic.CommonOperations;
using StackExchange.Redis;
using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using Farakonesh.Logic.Log;
using Farakonesh.Shared.Helpers;
using Microsoft.AspNetCore.Http;

namespace Farakonesh.Logic.Services.Cache
{
    public class RedisService : IDisposable
    {
        private readonly ILog _log;
        private readonly IHttpContextAccessor _httpContextAccessor;

        private IConnectionMultiplexer Connection;
        private IConnectionMultiplexer ConnectionAuxiliary;
        private IConnectionMultiplexer ConnectionExternalServer;

        private string _connectionString;
        private string _connectionStringAuxiliary;
        private string _connectionStringExternalServer;

        private bool _useExternalServer = false;

        public RedisService(IConfiguration configuration, ILog log, IHttpContextAccessor httpContextAccessor)
        {
            _log = log;
            _httpContextAccessor = httpContextAccessor;

            _connectionString = configuration.GetConnectionString("RedisCns");
            _connectionStringAuxiliary = configuration.GetConnectionString("RedisCnsAuxiliary");
            _connectionStringExternalServer = configuration.GetConnectionString("RedisCnsExternalServer");
        }

        public async Task InitAsync(string externalServerIp = null)
        {
            if (!string.IsNullOrWhiteSpace(externalServerIp))
            {
                var localIp = _httpContextAccessor.HttpContext?.Connection?.LocalIpAddress;

                var isExternal = await IpCheckHelper.IsExternalServerAsync(
                    externalServerIp,
                    new IpifyPublicIpProvider(),
                    localIp
                );
                if (isExternal)
                    _useExternalServer = true;
            }



            Connection = await ConnectionMultiplexer.ConnectAsync(_connectionString);
            ConnectionAuxiliary = await ConnectionMultiplexer.ConnectAsync(_connectionStringAuxiliary);
            ConnectionExternalServer = await ConnectionMultiplexer.ConnectAsync(_connectionStringExternalServer);

            Main = Connection.GetDatabase(1);
            MainAuxiliary = ConnectionAuxiliary.GetDatabase(1);
            MainExternalServer = ConnectionExternalServer.GetDatabase(1);
        }

        public StackExchange.Redis.IDatabase Main { get; private set; }
        public StackExchange.Redis.IDatabase MainAuxiliary { get; private set; }
        public StackExchange.Redis.IDatabase MainExternalServer { get; private set; }

        private (StackExchange.Redis.IDatabase db, IConnectionMultiplexer conn) GetRedisDatabase()
        {
            return _useExternalServer
                ? (MainExternalServer, ConnectionExternalServer)
                : (Main, Connection);
        }

        public async Task<T> Get<T>(string key)
        {
            var (db, conn) = GetRedisDatabase();

            if (conn == null || !conn.IsConnected)
                return default;

            var data = await db.StringGetAsync(key);
            if (!data.HasValue)
                return default;

            return data.JsonToT<T>();
        }

        public async Task<T> Set<T>(string key, object data, int expireMinute = 720)
        {
            var (db, conn) = GetRedisDatabase();

            var dataJson = data.ObjToJson();
            if (conn == null || !conn.IsConnected)
                return dataJson.JsonToT<T>();

            var expiry = expireMinute <= 0 ? (TimeSpan?)null : TimeSpan.FromMinutes(expireMinute);
            await db.StringSetAsync(key, dataJson, expiry);

            return dataJson.JsonToT<T>();
        }

        public async Task<bool> Remove(string key)
        {
            if (Connection == null || !Connection.IsConnected)
                return false;

            await Main.KeyDeleteAsync(key);
            return true;
        }

        public async Task<List<string>> Clear()
        {
            var list = new List<string>();

            try
            {
                if (Connection != null && Connection.IsConnected)
                {
                    var server = Connection.GetServer(Connection.GetEndPoints()[0]);
                    server.FlushAllDatabases();
                }
            }
            catch (Exception ex)
            {
                _log.Error($"Error clearing main Redis: {ex.ObjToJson()}");
                list.Add("خطا در پاک سازی کش دیتابیس سرور اصلی");
            }

            try
            {
                if (ConnectionAuxiliary != null && ConnectionAuxiliary.IsConnected)
                {
                    var server = ConnectionAuxiliary.GetServer(ConnectionAuxiliary.GetEndPoints()[0]);
                    server.FlushAllDatabases();
                }
            }
            catch (Exception ex)
            {
                _log.Error($"Error clearing auxiliary Redis: {ex.ObjToJson()}");
                list.Add("خطا در پاک سازی کش دیتابیس سرور کمکی");
            }

            try
            {
                if (ConnectionExternalServer != null && ConnectionExternalServer.IsConnected)
                {
                    var server = ConnectionExternalServer.GetServer(ConnectionExternalServer.GetEndPoints()[0]);
                    server.FlushAllDatabases();
                }
            }
            catch (Exception ex)
            {
                _log.Error($"Error clearing external Redis: {ex.ObjToJson()}");
                list.Add("خطا در پاک سازی کش دیتابیس سرور خارج");
            }

            return list;
        }

        public void Dispose()
        {
            Connection?.Dispose();
            ConnectionAuxiliary?.Dispose();
            ConnectionExternalServer?.Dispose();
        }
    }
}
