﻿using Microsoft.AspNetCore.Http;
using System.Collections.Generic;
using System.Linq;
using System.Security.Claims;
using System.Text;
using System;
using Microsoft.IdentityModel.Tokens;
using System.IdentityModel.Tokens.Jwt;
using Microsoft.AspNetCore.Authentication.JwtBearer;
using System.Threading.Tasks;
using Farakonesh.Shared.Exceptions;
using Farakonesh.Shared.Helpers;

namespace Farakonesh.ReportViewver.WebToken
{
    public class JWT
    {
        public JWT()
        {
        }
        private readonly UserInfo _user;
        public UserInfo User
        {
            get
            {
                return _user;
            }
        }

        public static readonly string issuer = "https://farakonesh.org";
        public static readonly string key = "F@@@@@@@@@***&&**!!!!@@@rakonesh2023JWT";
        public JWT(string token)
        {
            this._user = GetData(token);
        }
        public JWT(string token, bool isRefresh = false)
        {
            if (isRefresh == false)
                this._user = GetData(token);
            else
                this._user = GetDataRefresh(token);
        }
        /// <summary>
        /// Generate token validation
        /// </summary>
        /// <param name="key"></param>
        /// <param name="issuer"></param>
        /// <returns></returns>
        public static TokenValidationParameters GetTokenValidationParameters()
        {
            if (string.IsNullOrWhiteSpace(key) || string.IsNullOrWhiteSpace(issuer))
                throw new ArgumentNullException("key and issuer could not be empty. Check application settings.");

            var rst = new TokenValidationParameters
            {
                ValidateIssuer = true,
                ValidateAudience = true,
                ValidateIssuerSigningKey = true,
                ValidIssuer = issuer,
                ValidAudience = issuer,
                IssuerSigningKey = new SymmetricSecurityKey(Encoding.UTF8.GetBytes(key))
            };

            return rst;
        }

        public static WebToken.UserInfo getUserModel()
        {
            try
            {
                var userModel = new JWT(GetJWTToken());
                if (userModel == null || userModel.User == null)
                    return new WebToken.UserInfo();

                return userModel.User;
            }
            catch (Exception ex)
            {
                throw new InternalServiceException(ex.Message);
            }
        }

        public static WebToken.UserInfo GetUserModelRefresh(string token)
        {
            try
            {
                var userModel = new WebToken.JWT(token, true);
                if (userModel == null)
                    return new WebToken.UserInfo();

                return userModel.User;
            }
            catch (Exception ex)
            {
                throw new InternalServiceException(ex.Message);
            }
        }

        private static string GetClaimsFromContext(ClaimsPrincipal user, string key)
        {
            if (user == null)
                return null;
            var item = user.FindFirst(key);
            if (item == null)
                return null;
            return item.Value;
        }

        private UserInfo GetData(string token)
        {
            if (string.IsNullOrWhiteSpace(token))
                return null;
            var validateToken = GetPrincipal(token);

            if (validateToken != null)
            {
                var model = new UserInfo();
                if (validateToken.Identity != null && !string.IsNullOrWhiteSpace(validateToken.Identity.Name))
                {
                    model.Token = validateToken.Identity.Name;
                    var UserId = getValueFromClaim(validateToken.Claims, "UserId");
                    if (UserId != null)
                        model.UserId = UserId;
                    model.AccessToken = "Bearer "+token;
                    return model;
                }
            }

            return null;
        }

        private UserInfo GetDataRefresh(string token)
        {
            var validateToken = GetPrincipal(token);

            if (validateToken != null)
            {
                var model = new UserInfo();
                if (validateToken.Identity != null && !string.IsNullOrWhiteSpace(validateToken.Identity.Name))
                {
                    model.RefreshToken = validateToken.Identity.Name;
                    var UserId = getValueFromClaim(validateToken.Claims, "UserId");
                    if (UserId != null)
                        model.UserId = UserId;
                    return model;
                }
            }

            return null;
        }

        private ClaimsPrincipal GetPrincipal(string token)
        {
            var tokenHandler = new JwtSecurityTokenHandler();
            var jwtToken = tokenHandler.ReadToken(token) as JwtSecurityToken;

            if (jwtToken == null)
                return null;

            var symmetricKey = Encoding.UTF8.GetBytes(key);

            var validationParameters = new TokenValidationParameters()
            {
                RequireExpirationTime = false,
                ValidateIssuer = true,
                ValidateAudience = false,
                ValidateLifetime = false,
                ValidIssuer = issuer,
                IssuerSigningKey = new SymmetricSecurityKey(symmetricKey)
            };

            var principal = tokenHandler.ValidateToken(token, validationParameters, out _);

            return principal;
        }

        private string getValueFromClaim(IEnumerable<Claim> claims, string name)
        {
            try
            {
                var item = claims.FirstOrDefault(a => a.Type.ToLower().Equals(name.ToLower()));
                return (item.Value);
            }
            catch (Exception)
            {
                return null;
            }
        }
        public static string GetJWTToken()
        {
            try
            {
                IHttpContextAccessor ctx = new HttpContextAccessor();
                if (ctx.HttpContext == null)
                {
                    throw new InternalServiceException("Context Is Null");
                }
                var headers = ctx.HttpContext.Request.Query["Key"];
                if (headers.Count == 0)
                    //در صورتی که نتوانستی اطلاعات را بدست بیاوری، با حروف کوچک امتحان کن
                    headers = ctx.HttpContext.Request.Query["key"];

                if (headers.Count == 0)
                    return null;

                var header = headers.FirstOrDefault();
                string token = "";
                if (header != null)
                {
                    token = header.Replace("Bearer ", "").Replace("bearer ", "");
                }

                return (token);
            }
            catch (Exception ex)
            {
                throw new InternalServiceException(ex.Message);
            }
        }



        /// <summary>
        /// Get authentication event handler by JWT
        /// </summary>
        /// <returns></returns>
        public static JwtBearerEvents GetJWTEvents()
        {
            var rst = new JwtBearerEvents
            {
                OnAuthenticationFailed = context =>
                {
                    if (context.Exception.GetType() == typeof(SecurityTokenExpiredException))
                        context.Response.Headers.Add("Token-Expired", "true");

                    return Task.CompletedTask;
                }
            };

            return rst;
        }


        /// <summary>
        /// Validating input params
        /// </summary>
        /// <param name="claims"></param>
        /// <param name="key"></param>
        /// <param name="issuer"></param>
        /// <param name="expireDate"></param>
        private static void Validation(Claim[] claims, DateTime expireDate)
        {
            if (claims == null || claims.Count() == 0)
                throw new ArgumentNullException("In GenerateJWT, claims could not be null.");

            if (string.IsNullOrWhiteSpace(key))
                throw new ArgumentNullException("In GenerateJWT, key could not be empty.");

            if (string.IsNullOrWhiteSpace(issuer))
                throw new ArgumentNullException("In GenerateJWT, issuer could not be empty.");

            if (expireDate < DateTimeHelper.GetLocalTime())
                throw new ArgumentException("In GenerateJWT, expireDate could not be before the current date.");
        }

        /// <summary>
        /// Generate JWT based on the claims and security parameters
        /// </summary>
        /// <param name="claims"></param>
        /// <param name="key"></param>
        /// <param name="issuer"></param>
        /// <param name="expireDate"></param>
        /// <returns></returns>
        public static string Generate(Claim[] claims, DateTime expireDate)
        {
            Validation(claims, expireDate);

            var securityKey = new SymmetricSecurityKey(Encoding.UTF8.GetBytes(key));
            var credentials = new SigningCredentials(securityKey, SecurityAlgorithms.HmacSha256);

            JwtSecurityToken token = new JwtSecurityToken
            (
                issuer,     //Issure  
                issuer,     //Audience
                claims,
                expires: expireDate,
                signingCredentials: credentials
            );

            var jwt = new JwtSecurityTokenHandler().WriteToken(token);
            return jwt;
        }
    }
}
