﻿using Farakonesh.Shared.Validations;
using Microsoft.AspNetCore.Mvc.Filters;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using Ganss.Xss;
using Microsoft.AspNetCore.Mvc;
using AngleSharp.Dom;
using HtmlAgilityPack;
using System.Net;

namespace Farakonesh.Shared.Filters
{
    public class SanitizeHtmlFilter : IActionFilter
    {
        public void OnActionExecuting(ActionExecutingContext context)
        {
            var sanitizer = new HtmlSanitizer();

            sanitizer.AllowedTags.Clear();
            sanitizer.AllowedTags.Add("a");
            sanitizer.AllowedTags.Add("div");
            sanitizer.AllowedTags.Add("img");
            sanitizer.AllowedTags.Add("b");
            sanitizer.AllowedTags.Add("br");
            sanitizer.AllowedTags.Add("i");
            sanitizer.AllowedTags.Add("u");
            sanitizer.AllowedTags.Add("p");
            sanitizer.AllowedTags.Add("ul");
            sanitizer.AllowedTags.Add("li");
            sanitizer.AllowedTags.Add("strong");
            sanitizer.AllowedTags.Add("em");
            sanitizer.AllowedTags.Add("span");
            sanitizer.AllowedTags.Add("h1");
            sanitizer.AllowedTags.Add("h2");
            sanitizer.AllowedTags.Add("h3");
            sanitizer.AllowedTags.Add("h4");
            sanitizer.AllowedTags.Add("h5");
            sanitizer.AllowedTags.Add("h6");
            sanitizer.AllowedTags.Add("table");
            sanitizer.AllowedTags.Add("thead");
            sanitizer.AllowedTags.Add("tbody");
            sanitizer.AllowedTags.Add("tr");
            sanitizer.AllowedTags.Add("td");
            sanitizer.AllowedTags.Add("th");
            sanitizer.AllowedTags.Add("pre");
            sanitizer.AllowedTags.Add("code");
            sanitizer.AllowedTags.Add("blockquote");
            sanitizer.AllowedTags.Add("font");

            sanitizer.AllowedAttributes.Clear();
            sanitizer.AllowedAttributes.Add("href");
            sanitizer.AllowedAttributes.Add("src");
            sanitizer.AllowedAttributes.Add("alt");
            sanitizer.AllowedAttributes.Add("title");
            sanitizer.AllowedAttributes.Add("target");
            sanitizer.AllowedAttributes.Add("rel");
            sanitizer.AllowedAttributes.Add("style");
            sanitizer.AllowedAttributes.Add("class");
            sanitizer.AllowedAttributes.Add("color");
            sanitizer.AllowedAttributes.Add("colspan");
            sanitizer.AllowedAttributes.Add("rowspan");
            sanitizer.AllowedAttributes.Add("align");

            sanitizer.AllowedSchemes.Clear();
            sanitizer.AllowedSchemes.Add("http");
            sanitizer.AllowedSchemes.Add("https");

            sanitizer.AllowCssCustomProperties = false;
            sanitizer.AllowedCssProperties.Add("color");
            sanitizer.AllowedCssProperties.Add("font-weight");
            sanitizer.AllowedCssProperties.Add("text-align");

            sanitizer.PostProcessNode += (sender, e) =>
            {
                if (e.Node is IElement element)
                {
                    foreach (var attr in element.Attributes.ToList())
                    {
                        var attrName = attr.Name.ToLowerInvariant();
                        var attrValue = attr.Value?.ToLowerInvariant();

                        if (attrName.StartsWith("on"))
                        {
                            element.RemoveAttribute(attr.Name);
                            continue;
                        }


                        if ((attrName == "href" || attrName == "src") &&
                            (attrValue.StartsWith("javascript:") ||
                             attrValue.StartsWith("data:") ||
                             attrValue.Contains("base64") ||
                             attrValue.Contains("evil.com")))
                        {
                            element.RemoveAttribute(attr.Name);
                        }
                    }
                }
            };

            foreach (var arg in context.ActionArguments.Values)
            {
                if (arg == null) continue;

                var props = arg.GetType().GetProperties(BindingFlags.Public | BindingFlags.Instance);
                bool shouldSanitize = props.Any(p =>
                    p.PropertyType == typeof(string) &&
                    p.GetCustomAttribute<SanitizeHtmlAttribute>() != null &&
                    p.CanWrite);

                if (!shouldSanitize)
                    continue;


                foreach (var prop in props)
                {
                    if (prop.PropertyType == typeof(string) &&
                        prop.CanWrite &&
                        prop.GetCustomAttribute<SanitizeHtmlAttribute>() != null)
                    {
                        var original = prop.GetValue(arg) as string;
                        if (string.IsNullOrWhiteSpace(original)) continue;

                        var decoded = WebUtility.HtmlDecode(original);
                        var sanitized = sanitizer.Sanitize(decoded);
                        sanitized = ScrubDangerousTags(sanitized);

                        if (ContainsDangerousUrl(sanitized))
                        {
                            context.Result = new BadRequestObjectResult("متن شامل لینک یا URL خطرناک است.");
                            return;
                        }

                        prop.SetValue(arg, sanitized);
                    }
                }
            }
        }

        public void OnActionExecuted(ActionExecutedContext context)
        {
        }


        private string ScrubDangerousTags(string html)
        {
            var doc = new HtmlDocument();
            doc.LoadHtml(html);

            var nodes = doc.DocumentNode.SelectNodes("//*");
            if (nodes == null) return html;

            foreach (var node in nodes.ToList())
            {
                bool removeNode = false;

                foreach (var attr in node.Attributes.ToList())
                {
                    var name = attr.Name.ToLowerInvariant();
                    var value = attr.Value?.ToLowerInvariant();

                    if (name.StartsWith("on"))
                    {
                        removeNode = true;
                        break;
                    }

                    if ((name == "href" || name == "src") &&
                        (value.StartsWith("javascript:") ||
                         value.StartsWith("data:") ||
                         value.Contains("base64") ||
                         value.Contains("evil.com")))
                    {
                        removeNode = true;
                        break;
                    }

                    if (name == "style")
                    {
                        if (
                            value.Contains("position") ||
                            value.Contains("z-index") ||
                            value.Contains("fixed") ||
                            value.Contains("absolute") ||
                            value.Contains("display:flex") ||
                            value.Contains("display: flex") ||
                            value.Contains("top") ||
                            value.Contains("left") ||
                            value.Contains("right") ||
                            value.Contains("bottom") ||
                            value.Contains("width") ||
                            value.Contains("height") ||
                            value.Contains("overflow")
                        )
                        {
                            removeNode = true;
                            break;
                        }
                    }
                }

                if (removeNode)
                {
                    node.Remove();
                }
            }

            return doc.DocumentNode.InnerHtml;
        }

        private bool ContainsDangerousUrl(string html)
        {
            var doc = new HtmlAgilityPack.HtmlDocument();
            doc.LoadHtml(html);

            var tagsWithUrls = doc.DocumentNode.SelectNodes("//*[@href or @src]");
            if (tagsWithUrls == null) return false;

            foreach (var node in tagsWithUrls)
            {
                var url = node.GetAttributeValue("href", null) ?? node.GetAttributeValue("src", null);
                if (url == null) continue;

                url = url.ToLowerInvariant();

                if (
                    url.Contains("javascript:") ||
                    url.Contains("data:") ||
                    url.Contains("evil.com") ||
                    url.Contains("base64,")
                )
                {
                    return true;
                }
            }

            return false;
        }
    }
}
